import crypto from 'node:crypto'

const COOKIE_NAME = 'hls_admin_session'

const normalizeValue = (value) => (value ?? '').trim()

const timingSafeEqual = (left, right) => {
  const leftBuffer = Buffer.from(left)
  const rightBuffer = Buffer.from(right)
  if (leftBuffer.length !== rightBuffer.length) {
    return false
  }
  return crypto.timingSafeEqual(leftBuffer, rightBuffer)
}

const parseCookie = (header, name) => {
  if (!header) {
    return undefined
  }
  const parts = header.split(';').map((part) => part.trim())
  for (const part of parts) {
    if (!part.startsWith(`${name}=`)) {
      continue
    }
    return part.slice(name.length + 1)
  }
  return undefined
}

const isSecureRequest = (req) => {
  if (req.secure) {
    return true
  }
  const forwardedProto = req.get('x-forwarded-proto')
  if (!forwardedProto) {
    return false
  }
  return forwardedProto.split(',')[0]?.trim() === 'https'
}

const resolveHostname = (value) => {
  if (!value) return undefined
  try {
    return new URL(value).hostname
  } catch {
    return value.split(':')[0]
  }
}

const isCrossSiteRequest = (req) => {
  const origin = req.get('origin')
  if (!origin) {
    return false
  }
  const originHost = resolveHostname(origin)
  const hostHeader = req.get('host')
  const requestHost = resolveHostname(`http://${hostHeader}`)
  return Boolean(originHost && requestHost && originHost !== requestHost)
}

const createMemorySessionStore = () => {
  const sessions = new Map()
  const nowSeconds = () => Math.floor(Date.now() / 1000)

  const get = async (token) => {
    const session = sessions.get(token)
    if (!session) {
      return { status: 'missing' }
    }
    if (session.expiresAt && session.expiresAt <= nowSeconds()) {
      sessions.delete(token)
      return { status: 'expired' }
    }
    return { status: 'active', session: { token, ...session } }
  }

  const create = async ({ token, ttlSeconds }) => {
    const createdAt = nowSeconds()
    const expiresAt = ttlSeconds > 0 ? createdAt + ttlSeconds : null
    sessions.set(token, { createdAt, expiresAt })
    return { token, createdAt, expiresAt }
  }

  const touch = async (token, ttlSeconds) => {
    if (!ttlSeconds || ttlSeconds <= 0) {
      return false
    }
    const session = sessions.get(token)
    if (!session) {
      return false
    }
    session.expiresAt = nowSeconds() + ttlSeconds
    return true
  }

  const revoke = async (token) => {
    return sessions.delete(token)
  }

  const clear = async () => {
    sessions.clear()
  }

  return {
    get,
    create,
    touch,
    revoke,
    clear,
  }
}

export const createAdminAuth = ({
  username,
  password,
  sessionTtlSeconds,
  frontendOrigin,
  allowedOrigins = [],
  allowedHosts = [],
  sessionStore,
}) => {
  const normalizedUsername = normalizeValue(username)
  const normalizedPassword = normalizeValue(password)
  const hasCredentials = Boolean(normalizedUsername && normalizedPassword)
  const ttlSeconds = Number.isFinite(sessionTtlSeconds) ? sessionTtlSeconds : 0
  const cookieMaxAgeSeconds = ttlSeconds > 0 ? ttlSeconds : 10 * 365 * 24 * 60 * 60
  const store = sessionStore || createMemorySessionStore()

  const originSet = new Set(allowedOrigins.filter(Boolean))
  const hostSet = new Set(allowedHosts.filter(Boolean))

  const isOriginAllowed = (origin) => {
    if (!origin) {
      return false
    }
    if (originSet.has(origin)) {
      return true
    }
    try {
      const hostname = new URL(origin).hostname
      return hostSet.has(hostname)
    } catch {
      return false
    }
  }

  const createSession = () => {
    const token = crypto.randomBytes(32).toString('hex')
    return store.create({ token, ttlSeconds })
  }

  const getSession = async (req) => {
    const token = parseCookie(req.headers.cookie, COOKIE_NAME)
    if (!token) {
      return undefined
    }
    const result = await store.get(token)
    if (result.status !== 'active') {
      return undefined
    }
    if (ttlSeconds > 0) {
      await store.touch(token, ttlSeconds)
    }
    return result.session
  }

  const setSessionCookie = (res, token, req) => {
    const sameSite = isCrossSiteRequest(req) ? 'None' : 'Lax'
    const parts = [
      `${COOKIE_NAME}=${token}`,
      'Path=/',
      'HttpOnly',
      `SameSite=${sameSite}`,
      `Max-Age=${cookieMaxAgeSeconds}`,
    ]
    if (isSecureRequest(req) || sameSite === 'None') {
      parts.push('Secure')
    }
    res.setHeader('Set-Cookie', parts.join('; '))
  }

  const clearSessionCookie = (res) => {
    const parts = [
      `${COOKIE_NAME}=`,
      'Path=/',
      'HttpOnly',
      'SameSite=Strict',
      'Max-Age=0',
    ]
    res.setHeader('Set-Cookie', parts.join('; '))
  }

  const cors = (req, res, next) => {
    const origin = req.get('origin')
    let allowOrigin = null
    if (origin && isOriginAllowed(origin)) {
      allowOrigin = origin
    } else if (!originSet.size && !hostSet.size && origin) {
      allowOrigin = origin
    } else if (frontendOrigin) {
      allowOrigin = frontendOrigin
    }

    if (allowOrigin) {
      res.setHeader('Access-Control-Allow-Origin', allowOrigin)
      res.setHeader('Access-Control-Allow-Credentials', 'true')
      res.setHeader('Vary', 'Origin')
    }
    res.setHeader('Access-Control-Allow-Headers', 'Content-Type')
    res.setHeader('Access-Control-Allow-Methods', 'GET, POST, PATCH, DELETE, OPTIONS')
    if (req.method === 'OPTIONS') {
      res.status(204).end()
      return
    }
    next()
  }

  const login = async (req, res) => {
    if (!hasCredentials) {
      res.status(503).json({ error: 'admin_not_configured' })
      return
    }
    const body = req.body || {}
    const inputUsername = normalizeValue(body.username)
    const inputPassword = normalizeValue(body.password)

    const usernameOk = timingSafeEqual(inputUsername, normalizedUsername)
    const passwordOk = timingSafeEqual(inputPassword, normalizedPassword)

    if (!usernameOk || !passwordOk) {
      res.status(401).json({ error: 'invalid_credentials' })
      return
    }

    await store.clear()
    const session = await createSession()
    setSessionCookie(res, session.token, req)
    res.json({ authenticated: true, expiresAt: session.expiresAt || null })
  }

  const logout = async (_req, res) => {
    await store.clear()
    clearSessionCookie(res)
    res.json({ authenticated: false })
  }

  const me = async (req, res) => {
    if (!hasCredentials) {
      res.json({ authenticated: false, configured: false })
      return
    }
    const session = await getSession(req)
    res.json({ authenticated: Boolean(session), configured: true })
  }

  const requireAdmin = async (req, res, next) => {
    if (!hasCredentials) {
      res.status(503).json({ error: 'admin_not_configured' })
      return
    }
    const session = await getSession(req)
    if (!session) {
      res.status(401).json({ error: 'admin_required' })
      return
    }
    next()
  }

  return {
    cors,
    login,
    logout,
    me,
    requireAdmin,
    isConfigured: hasCredentials,
  }
}
