import fs from 'node:fs/promises'
import path from 'node:path'

const DEFAULT_STORE = { sessions: {} }
const nowSeconds = () => Math.floor(Date.now() / 1000)

export const createAdminSessionStore = ({ filePath }) => {
  let cache = null
  let loadPromise = null
  let writeQueue = Promise.resolve()

  const ensureDir = async () => {
    await fs.mkdir(path.dirname(filePath), { recursive: true })
  }

  const load = async () => {
    if (cache) {
      return cache
    }
    if (!loadPromise) {
      loadPromise = (async () => {
        try {
          const raw = await fs.readFile(filePath, 'utf8')
          cache = JSON.parse(raw)
        } catch (error) {
          if (error.code !== 'ENOENT') {
            throw error
          }
          cache = { ...DEFAULT_STORE }
        }
        if (!cache || typeof cache !== 'object' || !cache.sessions) {
          cache = { ...DEFAULT_STORE }
        }
        return cache
      })()
    }
    return loadPromise
  }

  const persist = async () => {
    await ensureDir()
    const payload = JSON.stringify(cache, null, 2)
    writeQueue = writeQueue.then(() => fs.writeFile(filePath, payload))
    await writeQueue
  }

  const pruneExpired = (store) => {
    const now = nowSeconds()
    let changed = false
    for (const [token, session] of Object.entries(store.sessions)) {
      if (session.expiresAt && session.expiresAt <= now) {
        delete store.sessions[token]
        changed = true
      }
    }
    return changed
  }

  const get = async (token) => {
    const store = await load()
    const changed = pruneExpired(store)
    if (changed) {
      await persist()
    }
    const session = store.sessions[token]
    if (!session) {
      return { status: 'missing' }
    }
    if (session.expiresAt && session.expiresAt <= nowSeconds()) {
      delete store.sessions[token]
      await persist()
      return { status: 'expired' }
    }
    return { status: 'active', session: { token, ...session } }
  }

  const create = async ({ token, ttlSeconds }) => {
    const store = await load()
    const createdAt = nowSeconds()
    const expiresAt = ttlSeconds > 0 ? createdAt + ttlSeconds : null
    store.sessions[token] = {
      createdAt,
      expiresAt,
    }
    await persist()
    return { token, ...store.sessions[token] }
  }

  const touch = async (token, ttlSeconds) => {
    if (!ttlSeconds || ttlSeconds <= 0) {
      return false
    }
    const store = await load()
    const session = store.sessions[token]
    if (!session) {
      return false
    }
    session.expiresAt = nowSeconds() + ttlSeconds
    await persist()
    return true
  }

  const revoke = async (token) => {
    const store = await load()
    if (!store.sessions[token]) {
      return false
    }
    delete store.sessions[token]
    await persist()
    return true
  }

  const clear = async () => {
    const store = await load()
    store.sessions = {}
    await persist()
  }

  return {
    get,
    create,
    touch,
    revoke,
    clear,
  }
}
