import fs from 'node:fs/promises'
import path from 'node:path'

const DEFAULT_STORE = { tokens: {} }

const nowSeconds = () => Math.floor(Date.now() / 1000)

export const createEmbedTokenStore = ({ filePath }) => {
  let cache = null
  let loadPromise = null
  let writeQueue = Promise.resolve()

  const ensureDir = async () => {
    await fs.mkdir(path.dirname(filePath), { recursive: true })
  }

  const load = async () => {
    if (cache) {
      return cache
    }
    if (!loadPromise) {
      loadPromise = (async () => {
        try {
          const raw = await fs.readFile(filePath, 'utf8')
          cache = JSON.parse(raw)
        } catch (error) {
          if (error.code !== 'ENOENT') {
            throw error
          }
          cache = { ...DEFAULT_STORE }
        }
        if (!cache || typeof cache !== 'object' || !cache.tokens) {
          cache = { ...DEFAULT_STORE }
        }
        return cache
      })()
    }
    return loadPromise
  }

  const persist = async () => {
    await ensureDir()
    const payload = JSON.stringify(cache, null, 2)
    writeQueue = writeQueue.then(() => fs.writeFile(filePath, payload))
    await writeQueue
  }

  const pruneExpired = (store) => {
    const now = nowSeconds()
    let changed = false
    for (const [tokenId, token] of Object.entries(store.tokens)) {
      if (token.expiresAt && token.expiresAt <= now) {
        delete store.tokens[tokenId]
        changed = true
      }
    }
    return changed
  }

  const list = async () => {
    const store = await load()
    const changed = pruneExpired(store)
    if (changed) {
      await persist()
    }
    return Object.entries(store.tokens).map(([tokenId, token]) => ({
      tokenId,
      ...token,
    }))
  }

  const get = async (tokenId) => {
    const store = await load()
    const token = store.tokens[tokenId]
    if (!token) {
      return { status: 'missing' }
    }
    if (token.expiresAt && token.expiresAt <= nowSeconds()) {
      delete store.tokens[tokenId]
      await persist()
      return { status: 'expired' }
    }
    return { status: 'active', token: { tokenId, ...token } }
  }

  const create = async ({ tokenId, config, ttlSeconds, description }) => {
    const store = await load()
    const createdAt = nowSeconds()
    const expiresAt = typeof ttlSeconds === 'number' ? createdAt + ttlSeconds : null
    store.tokens[tokenId] = {
      config,
      createdAt,
      expiresAt,
      ttlSeconds: typeof ttlSeconds === 'number' ? ttlSeconds : null,
      usageCount: 0,
      lastAccessedAt: null,
      description: description || null,
    }
    await persist()
    return { tokenId, ...store.tokens[tokenId] }
  }

  const revoke = async (tokenId) => {
    const store = await load()
    if (!store.tokens[tokenId]) {
      return false
    }
    delete store.tokens[tokenId]
    await persist()
    return true
  }

  const recordUsage = async (tokenId) => {
    const store = await load()
    const token = store.tokens[tokenId]
    if (!token) {
      return { status: 'missing' }
    }
    if (token.expiresAt && token.expiresAt <= nowSeconds()) {
      delete store.tokens[tokenId]
      await persist()
      return { status: 'expired' }
    }
    token.usageCount = (token.usageCount || 0) + 1
    token.lastAccessedAt = nowSeconds()
    await persist()
    return { status: 'recorded', token: { tokenId, ...token } }
  }

  const prune = async () => {
    const store = await load()
    const changed = pruneExpired(store)
    if (changed) {
      await persist()
    }
    return changed
  }

  return {
    list,
    get,
    create,
    revoke,
    recordUsage,
    pruneExpired: prune,
  }
}
