import { spawn } from 'node:child_process'
import fs from 'node:fs/promises'
import path from 'node:path'

const DEFAULT_SEGMENT_TIME = 4

export const buildCopyArgs = ({ input, outputDir }) => [
  '-hide_banner',
  '-y',
  '-i', input,
  '-c', 'copy',
  '-hls_time', String(DEFAULT_SEGMENT_TIME),
  '-hls_playlist_type', 'vod',
  '-hls_flags', 'independent_segments',
  '-hls_segment_filename', path.join(outputDir, 'segment-%03d.ts'),
  path.join(outputDir, 'index.m3u8'),
]

export const buildTranscodeArgs = ({
  input,
  outputDir,
  target,
  useVideoToolbox,
}) => {
  const videoCodec = target === 'hevc'
    ? (useVideoToolbox ? 'hevc_videotoolbox' : 'libx265')
    : (useVideoToolbox ? 'h264_videotoolbox' : 'libx264')

  return [
    '-hide_banner',
    '-y',
    '-i', input,
    '-c:v', videoCodec,
    '-c:a', 'aac',
    '-b:a', '192k',
    '-pix_fmt', 'yuv420p',
    '-hls_time', String(DEFAULT_SEGMENT_TIME),
    '-hls_playlist_type', 'vod',
    '-hls_flags', 'independent_segments',
    '-hls_segment_filename', path.join(outputDir, 'segment-%03d.ts'),
    path.join(outputDir, 'index.m3u8'),
  ]
}

const runProcess = (command, args) => new Promise((resolve, reject) => {
  const child = spawn(command, args)
  let stderr = ''

  child.stderr.on('data', (chunk) => {
    stderr += chunk.toString()
  })

  child.on('error', (error) => {
    reject(error)
  })

  child.on('close', (code) => {
    if (code === 0) {
      resolve()
      return
    }
    const error = new Error(`ffmpeg_failed:${code}`)
    error.details = stderr
    reject(error)
  })
})

const ensureEmptyDir = async (dir) => {
  await fs.rm(dir, { recursive: true, force: true })
  await fs.mkdir(dir, { recursive: true })
}

const createQueue = (maxJobs) => {
  let running = 0
  const pending = []

  const runNext = () => {
    if (running >= maxJobs) {
      return
    }
    const next = pending.shift()
    if (!next) {
      return
    }
    running += 1
    next()
  }

  const enqueue = (task) => new Promise((resolve, reject) => {
    const execute = () => {
      task()
        .then(resolve)
        .catch(reject)
        .finally(() => {
          running -= 1
          runNext()
        })
    }
    pending.push(execute)
    runNext()
  })

  return { enqueue }
}

export const createHlsPackager = ({
  cacheRoot,
  ffmpegPath,
  transcodeTarget,
  maxJobs,
}) => {
  const inflight = new Map()
  const queue = createQueue(Math.max(1, maxJobs || 1))

  const buildOutputDir = (shareId) => path.join(cacheRoot, shareId)

  const ensureHls = async ({ shareId, sourcePath }) => {
    const outputDir = buildOutputDir(shareId)
    const indexPath = path.join(outputDir, 'index.m3u8')

    try {
      await fs.access(indexPath)
      return outputDir
    } catch {
      // continue
    }

    if (inflight.has(shareId)) {
      await inflight.get(shareId)
      return outputDir
    }

    const task = queue.enqueue(async () => {
      await ensureEmptyDir(outputDir)
      try {
        await runProcess(ffmpegPath, buildCopyArgs({ input: sourcePath, outputDir }))
      } catch (error) {
        if (transcodeTarget === 'copy') {
          throw error
        }
        const target = transcodeTarget === 'hevc' ? 'hevc' : 'h264'
        await runProcess(ffmpegPath, buildTranscodeArgs({
          input: sourcePath,
          outputDir,
          target,
          useVideoToolbox: false,
        }))
      }
    })

    inflight.set(shareId, task)
    try {
      await task
      return outputDir
    } finally {
      inflight.delete(shareId)
    }
  }

  return {
    ensureHls,
  }
}
