import path from 'node:path'
import fs from 'node:fs'
import fsPromises from 'node:fs/promises'
import { fileURLToPath } from 'node:url'
import { Readable, pipeline } from 'node:stream'

import express from 'express'
import cors from 'cors'
import { fetch } from 'undici'
import multer from 'multer'
import { v4 as uuidv4 } from 'uuid'
import helmet from 'helmet'
import rateLimit from 'express-rate-limit'

import { validateProxyUrl, rewritePlaylist } from './proxy.js'
import { createAdminAuth } from './adminAuth.js'
import { createAdminSessionStore } from './adminSessionStore.js'
import { createShareStore } from './localMediaStore.js'
import { createEmbedTokenStore } from './embedTokenStore.js'
import {
  detectKind,
  getMimeType,
  listEntries,
  resolveMediaRoot,
  resolveRelativePath,
  rewriteLocalPlaylist,
} from './localMedia.js'
import { createHlsPackager } from './hlsPackager.js'

const __filename = fileURLToPath(import.meta.url)
const __dirname = path.dirname(__filename)

const app = express()
const port = Number(process.env.BACKEND_PORT || 2738)
const host = process.env.BACKEND_HOST || '0.0.0.0'

const resolveOrigin = (value) => {
  try {
    return new URL(value).origin
  } catch {
    return undefined
  }
}

const parseNumber = (value, fallback) => {
  const parsed = Number(value)
  return Number.isFinite(parsed) ? parsed : fallback
}

const parseTrustProxy = (value) => {
  if (!value) {
    return false
  }
  const normalized = value.trim().toLowerCase()
  if (normalized === 'true') {
    return 1
  }
  if (normalized === 'false') {
    return false
  }
  if (normalized === 'loopback') {
    return 'loopback'
  }
  const numeric = Number(normalized)
  if (Number.isFinite(numeric)) {
    return numeric
  }
  return value
}

const parseList = (value) => {
  if (!value) {
    return []
  }
  return value
    .split(',')
    .map((item) => item.trim())
    .filter(Boolean)
}

const parseOrigins = (value) => parseList(value)
  .map((item) => resolveOrigin(item))
  .filter(Boolean)

const frontendUrl = process.env.FRONTEND_URL || process.env.FRONTEND_PUBLIC_URL || 'http://127.0.0.1:7594'
const frontendOrigins = [
  ...parseOrigins(process.env.FRONTEND_URL),
  ...parseOrigins(process.env.FRONTEND_PUBLIC_URL),
]
if (frontendOrigins.length === 0) {
  const fallbackOrigin = resolveOrigin(frontendUrl)
  if (fallbackOrigin) {
    frontendOrigins.push(fallbackOrigin)
  }
}
const frontendOrigin = frontendOrigins[0]
const frontendAllowedHosts = parseList(process.env.FRONTEND_ALLOWED_HOSTS)

const localMediaEnabled = process.env.ENABLE_LOCAL_MEDIA === '1'
const localMediaRoot = process.env.LOCAL_MEDIA_ROOT || ''
const localSharesPath = process.env.LOCAL_SHARES_DB_PATH || path.join(__dirname, 'data', 'local-shares.json')
const adminSessionsPath = process.env.ADMIN_SESSIONS_DB_PATH || path.join(__dirname, 'data', 'admin-sessions.json')
const localHlsCacheDir = process.env.LOCAL_HLS_CACHE_DIR || path.join(__dirname, 'data', 'hls-cache')
const localShareTtlSeconds = parseNumber(process.env.LOCAL_SHARE_TTL_SECONDS, 24 * 60 * 60)
const localShareMaxTtlSeconds = parseNumber(process.env.LOCAL_SHARE_MAX_TTL_SECONDS, 30 * 24 * 60 * 60)
const embedTokensPath = process.env.EMBED_TOKENS_DB_PATH || path.join(__dirname, 'data', 'embed-tokens.json')
const allowedTranscodeTargets = new Set(['auto', 'copy', 'h264', 'hevc'])
const localTranscodeTarget = allowedTranscodeTargets.has(process.env.LOCAL_TRANSCODE_TARGET)
  ? process.env.LOCAL_TRANSCODE_TARGET
  : 'auto'
const ffmpegPath = process.env.FFMPEG_PATH || 'ffmpeg'
const localMaxJobs = parseNumber(process.env.FFMPEG_MAX_JOBS, 1)
const localPackagerUrl = process.env.LOCAL_PACKAGER_URL
const localPackagerToken = process.env.LOCAL_PACKAGER_TOKEN

const adminSessionStore = createAdminSessionStore({ filePath: adminSessionsPath })
const shareStore = createShareStore({ filePath: localSharesPath })
const embedTokenStore = createEmbedTokenStore({ filePath: embedTokensPath })
const hlsPackager = createHlsPackager({
  cacheRoot: localHlsCacheDir,
  ffmpegPath,
  transcodeTarget: localTranscodeTarget,
  maxJobs: localMaxJobs,
})

const mediaRootPromise = localMediaEnabled ? resolveMediaRoot(localMediaRoot) : Promise.resolve(null)

const adminAuth = createAdminAuth({
  username: process.env.ADMIN_USERNAME,
  password: process.env.ADMIN_PASSWORD,
  sessionTtlSeconds: Number(process.env.ADMIN_SESSION_TTL_SECONDS || 0),
  frontendOrigin,
  allowedOrigins: frontendOrigins,
  allowedHosts: frontendAllowedHosts,
  sessionStore: adminSessionStore,
})

// Respect X-Forwarded-* headers from reverse proxies (Caddy/Nginx).
const trustProxyEnv = process.env.TRUST_PROXY
const trustProxy = trustProxyEnv === undefined ? 1 : parseTrustProxy(trustProxyEnv)
app.set('trust proxy', trustProxy)

// Security Hardening
app.use(helmet({
  contentSecurityPolicy: false, // Disabled for HLS playback compatibility
  crossOriginResourcePolicy: { policy: 'cross-origin' },
  frameguard: false, // Allow cross-site embedding (iframe)
}))

// Rate Limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // limit each IP to 100 requests per windowMs
  standardHeaders: true,
  legacyHeaders: false,
})
app.use(limiter)

app.use(express.json({ limit: '2mb' }))

app.use(cors({
  origin: true,
  credentials: true,
  methods: ['GET', 'POST', 'OPTIONS', 'PUT', 'DELETE', 'PATCH'],
  allowedHeaders: ['Content-Type', 'Range', 'Authorization', 'X-Requested-With'],
  exposedHeaders: ['Content-Length', 'Content-Range'],
}))

// File Upload Configuration
const uploadDir = path.join(__dirname, 'uploads')
const allowedUploadExtensions = new Set([
  '.jpeg',
  '.jpg',
  '.png',
  '.gif',
  '.webp',
  '.tiff',
  '.tif',
  '.bmp',
])
const allowedUploadMimeTypes = new Set([
  'image/jpeg',
  'image/png',
  'image/gif',
  'image/webp',
  'image/tiff',
  'image/bmp',
])
const storage = multer.diskStorage({
  destination: (_req, _file, cb) => {
    cb(null, uploadDir)
  },
  filename: (_req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase()
    cb(null, `${uuidv4()}${ext}`)
  }
})

const upload = multer({
  storage,
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB limit
  },
  fileFilter: (_req, file, cb) => {
    const extname = path.extname(file.originalname).toLowerCase()
    const isAllowedExt = allowedUploadExtensions.has(extname)
    const isAllowedMime = allowedUploadMimeTypes.has(file.mimetype)
    if (isAllowedExt && isAllowedMime) {
      return cb(null, true)
    }
    cb(new Error('Only image files are allowed!'))
  }
})

const getPublicBase = (req) => {
  const forwardedProto = req.get('x-forwarded-proto')
  const forwardedHost = req.get('x-forwarded-host')
  const protocol = forwardedProto ? forwardedProto.split(',')[0].trim() : req.protocol
  const hostHeader = forwardedHost ? forwardedHost.split(',')[0].trim() : req.get('host')
  return `${protocol}://${hostHeader}`
}

app.get('/uploads/list', async (req, res) => {
  try {
    await fsPromises.mkdir(uploadDir, { recursive: true })
    const entries = await fsPromises.readdir(uploadDir, { withFileTypes: true })
    const files = await Promise.all(entries
      .filter((entry) => entry.isFile())
      .map(async (entry) => {
        const absolute = path.join(uploadDir, entry.name)
        const extension = path.extname(entry.name).toLowerCase()
        if (!allowedUploadExtensions.has(extension)) {
          return null
        }
        let updatedAt = 0
        try {
          const stat = await fsPromises.stat(absolute)
          updatedAt = Math.floor(stat.mtimeMs / 1000)
        } catch {
          updatedAt = 0
        }
        return {
          name: entry.name,
          url: `${getPublicBase(req)}/uploads/${entry.name}`,
          updatedAt,
        }
      }))
    const filtered = files.filter(Boolean)
    filtered.sort((a, b) => b.updatedAt - a.updatedAt)
    res.json({ files: filtered })
  } catch (error) {
    res.status(500).json({ error: 'uploads_unavailable' })
  }
})

// Serve uploaded files statically
app.use('/uploads', express.static(uploadDir))

app.post('/upload', upload.single('file'), (req, res) => {
  if (!req.file) {
    return res.status(400).json({ error: 'No file uploaded' })
  }
  const fileUrl = `${getPublicBase(req)}/uploads/${req.file.filename}`
  res.json({ url: fileUrl })
})

app.get('/health', (_req, res) => {
  res.json({ status: 'ok' })
})

app.post('/admin/login', adminAuth.cors, adminAuth.login)
app.post('/admin/logout', adminAuth.cors, adminAuth.logout)
app.get('/admin/me', adminAuth.cors, adminAuth.me)

const embedTokenPresets = {
  '3h': 3 * 60 * 60,
  '6h': 6 * 60 * 60,
  '12h': 12 * 60 * 60,
  '24h': 24 * 60 * 60,
  '1d': 24 * 60 * 60,
  '3d': 3 * 24 * 60 * 60,
  '5d': 5 * 24 * 60 * 60,
  '7d': 7 * 24 * 60 * 60,
  '10d': 10 * 24 * 60 * 60,
  '14d': 14 * 24 * 60 * 60,
  '30d': 30 * 24 * 60 * 60,
}

const resolveEmbedTtlSeconds = ({ ttlPreset, customTtlValue, customTtlUnit }) => {
  if (ttlPreset && embedTokenPresets[ttlPreset]) {
    return embedTokenPresets[ttlPreset]
  }
  const numericValue = Number(customTtlValue)
  if (!Number.isFinite(numericValue) || numericValue <= 0) {
    return null
  }
  const unit = typeof customTtlUnit === 'string' ? customTtlUnit.toLowerCase() : 'hours'
  const multiplier = unit === 'days' ? 24 * 60 * 60 : 60 * 60
  return Math.floor(numericValue * multiplier)
}

const ensureLocalEnabled = async (req, res, next) => {
  if (!localMediaEnabled) {
    res.status(404).json({ error: 'local_disabled' })
    return
  }
  try {
    req.localMediaRoot = await mediaRootPromise
    if (!req.localMediaRoot) {
      res.status(500).json({ error: 'local_root_missing' })
      return
    }
    next()
  } catch (error) {
    res.status(500).json({ error: 'local_root_invalid' })
  }
}

const clampTtl = (ttl) => {
  if (!Number.isFinite(ttl)) {
    return localShareTtlSeconds
  }
  const bounded = Math.min(Math.max(ttl, 60), localShareMaxTtlSeconds)
  return bounded
}

const resolveShare = async (shareId, res) => {
  const result = await shareStore.get(shareId)
  if (result.status === 'missing') {
    res.status(404).json({ error: 'share_not_found' })
    return null
  }
  if (result.status === 'expired') {
    res.status(410).json({ error: 'share_expired' })
    return null
  }
  return result.share
}

app.get('/local/config', ensureLocalEnabled, (_req, res) => {
  res.json({
    enabled: true,
    defaultTtlSeconds: localShareTtlSeconds,
    maxTtlSeconds: localShareMaxTtlSeconds,
  })
})

app.post('/embed/tokens', adminAuth.cors, adminAuth.requireAdmin, async (req, res) => {
  const body = req.body || {}
  const ttlSeconds = resolveEmbedTtlSeconds({
    ttlPreset: body.ttlPreset,
    customTtlValue: body.customTtlValue,
    customTtlUnit: body.customTtlUnit,
  })

  if (!ttlSeconds) {
    res.status(400).json({ error: 'invalid_ttl' })
    return
  }

  if (!body.config || typeof body.config !== 'object') {
    res.status(400).json({ error: 'invalid_config' })
    return
  }

  const tokenId = uuidv4()
  const description = typeof body.description === 'string' ? body.description.trim() : ''
  const created = await embedTokenStore.create({
    tokenId,
    config: body.config,
    ttlSeconds,
    description,
  })

  res.json({
    tokenId: created.tokenId,
    expiresAt: created.expiresAt,
    createdAt: created.createdAt,
    ttlSeconds: created.ttlSeconds,
    description: created.description,
  })
})

app.get('/embed/tokens', adminAuth.cors, adminAuth.requireAdmin, async (_req, res) => {
  const tokens = await embedTokenStore.list()
  res.json({ tokens })
})

app.delete('/embed/tokens/:tokenId', adminAuth.cors, adminAuth.requireAdmin, async (req, res) => {
  const { tokenId } = req.params
  const removed = await embedTokenStore.revoke(tokenId)
  res.json({ removed })
})

app.get('/embed/token/:tokenId', async (req, res) => {
  const { tokenId } = req.params
  const result = await embedTokenStore.recordUsage(tokenId)
  if (result.status === 'missing') {
    res.status(404).json({ error: 'token_not_found' })
    return
  }
  if (result.status === 'expired') {
    res.status(410).json({ error: 'token_expired' })
    return
  }
  const token = result.token
  res.json({
    config: token.config,
    expiresAt: token.expiresAt,
    usageCount: token.usageCount,
    lastAccessedAt: token.lastAccessedAt,
  })
})

app.get('/local/browse', adminAuth.cors, adminAuth.requireAdmin, ensureLocalEnabled, async (req, res) => {
  const dir = typeof req.query.dir === 'string' ? req.query.dir : ''
  try {
    const entries = await listEntries(req.localMediaRoot, dir)
    res.json({ dir, entries })
  } catch (error) {
    res.status(400).json({ error: 'invalid_path' })
  }
})

app.get('/local/shares', adminAuth.cors, adminAuth.requireAdmin, ensureLocalEnabled, async (_req, res) => {
  const shares = await shareStore.list()
  res.json({ shares })
})

app.post('/local/shares', adminAuth.cors, adminAuth.requireAdmin, ensureLocalEnabled, async (req, res) => {
  const body = req.body || {}
  const relativePath = body.path
  const ttlSeconds = clampTtl(Number(body.ttlSeconds))
  const posterUrl = typeof body.posterUrl === 'string' ? body.posterUrl.trim() : null

  try {
    const resolved = await resolveRelativePath(req.localMediaRoot, relativePath)
    const stat = await fsPromises.stat(resolved.absolute)
    if (!stat.isFile()) {
      res.status(400).json({ error: 'not_a_file' })
      return
    }
    const kind = detectKind(resolved.absolute)
    const shareId = uuidv4()
    const created = await shareStore.create({
      shareId,
      path: resolved.relative,
      kind,
      ttlSeconds,
      posterUrl,
    })
    res.json({
      shareId: created.shareId,
      token: `${kind === 'hls' ? 'localhls' : 'localfile'}:${created.shareId}`,
      expiresAt: created.expiresAt,
      path: created.path,
      kind: created.kind,
      posterUrl: created.posterUrl || null,
    })
  } catch (error) {
    res.status(400).json({ error: 'invalid_path' })
  }
})

app.patch('/local/shares/:shareId', adminAuth.cors, adminAuth.requireAdmin, ensureLocalEnabled, async (req, res) => {
  const { shareId } = req.params
  const body = req.body || {}
  const posterUrl = typeof body.posterUrl === 'string'
    ? body.posterUrl.trim()
    : (typeof body.poster === 'string' ? body.poster.trim() : null)
  const result = await shareStore.update(shareId, { posterUrl })
  if (result.status === 'missing') {
    res.status(404).json({ error: 'share_not_found' })
    return
  }
  res.json({ share: result.share })
})

app.delete('/local/shares/:shareId', adminAuth.cors, adminAuth.requireAdmin, ensureLocalEnabled, async (req, res) => {
  const { shareId } = req.params
  const removed = await shareStore.revoke(shareId)
  res.json({ removed })
})

const pipeStream = (req, res, stream) => {
  const abort = () => {
    stream.destroy()
  }

  req.on('aborted', abort)
  res.on('close', abort)

  pipeline(stream, res, (error) => {
    req.off('aborted', abort)
    res.off('close', abort)
    if (error?.code === 'ERR_STREAM_PREMATURE_CLOSE') {
      return
    }
    if (error && !res.headersSent) {
      res.status(500).end()
    }
  })
}

const sendFileStream = async (req, res, absolutePath) => {
  const stat = await fsPromises.stat(absolutePath)
  const total = stat.size
  const mimeType = getMimeType(absolutePath)

  res.setHeader('Accept-Ranges', 'bytes')
  res.setHeader('Content-Type', mimeType)

  const range = req.headers.range
  if (!range) {
    res.setHeader('Content-Length', total)
    const stream = fs.createReadStream(absolutePath)
    pipeStream(req, res, stream)
    return
  }

  const match = /bytes=(\d*)-(\d*)/.exec(range)
  if (!match) {
    res.status(416).end()
    return
  }

  const start = match[1] ? Number(match[1]) : 0
  const end = match[2] ? Number(match[2]) : total - 1

  if (Number.isNaN(start) || Number.isNaN(end) || start > end || end >= total) {
    res.status(416).setHeader('Content-Range', `bytes */${total}`).end()
    return
  }

  res.status(206)
  res.setHeader('Content-Length', end - start + 1)
  res.setHeader('Content-Range', `bytes ${start}-${end}/${total}`)
  const stream = fs.createReadStream(absolutePath, { start, end })
  pipeStream(req, res, stream)
}

const ensureHlsCache = async ({ shareId, sourceAbsolute, sourceRelative }) => {
  const outputDir = path.join(localHlsCacheDir, shareId)
  const indexPath = path.join(outputDir, 'index.m3u8')
  try {
    await fsPromises.access(indexPath)
    return { outputDir, indexPath }
  } catch {
    // continue
  }

  if (localPackagerUrl) {
    if (!localPackagerToken) {
      throw new Error('packager_token_missing')
    }
    const response = await fetch(`${localPackagerUrl.replace(/\/$/, '')}/packager/hlsify`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        Authorization: `Bearer ${localPackagerToken}`,
      },
      body: JSON.stringify({
        shareId,
        relativePath: sourceRelative,
        target: localTranscodeTarget,
      }),
    })
    if (!response.ok) {
      throw new Error('packager_failed')
    }
  } else {
    await hlsPackager.ensureHls({ shareId, sourcePath: sourceAbsolute })
  }

  await fsPromises.access(indexPath)
  return { outputDir, indexPath }
}

app.get('/local/share/file/:shareId', ensureLocalEnabled, async (req, res) => {
  const { shareId } = req.params
  const share = await resolveShare(shareId, res)
  if (!share) {
    return
  }
  if (share.kind !== 'file') {
    res.status(400).json({ error: 'invalid_share_kind' })
    return
  }

  try {
    const resolved = await resolveRelativePath(req.localMediaRoot, share.path)
    await sendFileStream(req, res, resolved.absolute)
  } catch {
    res.status(404).json({ error: 'file_not_found' })
  }
})

app.get('/local/share/hls/:shareId/index.m3u8', ensureLocalEnabled, async (req, res) => {
  const { shareId } = req.params
  const share = await resolveShare(shareId, res)
  if (!share) {
    return
  }
  if (share.kind !== 'hls') {
    res.status(400).json({ error: 'invalid_share_kind' })
    return
  }

  try {
    const resolved = await resolveRelativePath(req.localMediaRoot, share.path)
    const ext = path.extname(resolved.absolute).toLowerCase()
    if (ext === '.m3u8') {
      const playlist = await fsPromises.readFile(resolved.absolute, 'utf8')
      const publicBase = getPublicBase(req)
      const rewritten = rewriteLocalPlaylist({
        playlistBody: playlist,
        playlistPath: resolved.absolute,
        shareId,
        rootReal: req.localMediaRoot,
        publicBase,
      })
      res.setHeader('Content-Type', 'application/vnd.apple.mpegurl')
      res.send(rewritten)
      return
    }

    const cache = await ensureHlsCache({
      shareId,
      sourceAbsolute: resolved.absolute,
      sourceRelative: resolved.relative,
    })
    const playlist = await fsPromises.readFile(cache.indexPath, 'utf8')
    res.setHeader('Content-Type', 'application/vnd.apple.mpegurl')
    res.send(playlist)
  } catch (error) {
    if (error.code === 'ENOENT') {
      res.status(404).json({ error: 'file_not_found' })
      return
    }
    res.status(500).json({ error: 'hls_failed' })
  }
})

app.get('/local/share/hls/:shareId/asset', ensureLocalEnabled, async (req, res) => {
  const { shareId } = req.params
  const share = await resolveShare(shareId, res)
  if (!share) {
    return
  }
  if (share.kind !== 'hls') {
    res.status(400).json({ error: 'invalid_share_kind' })
    return
  }

  const assetPath = typeof req.query.path === 'string' ? req.query.path : ''
  if (!assetPath) {
    res.status(400).json({ error: 'missing_asset_path' })
    return
  }

  try {
    const resolved = await resolveRelativePath(req.localMediaRoot, assetPath)
    await sendFileStream(req, res, resolved.absolute)
  } catch {
    res.status(404).json({ error: 'asset_not_found' })
  }
})

app.get('/local/share/hls/:shareId/:asset', ensureLocalEnabled, async (req, res) => {
  const { shareId, asset } = req.params
  const share = await resolveShare(shareId, res)
  if (!share) {
    return
  }
  if (share.kind !== 'hls') {
    res.status(400).json({ error: 'invalid_share_kind' })
    return
  }
  if (!asset || asset.includes('..') || asset.includes('/')) {
    res.status(400).json({ error: 'invalid_asset' })
    return
  }

  const outputDir = path.join(localHlsCacheDir, shareId)
  const assetPath = path.join(outputDir, asset)
  const relative = path.relative(outputDir, assetPath)
  if (relative.startsWith('..') || path.isAbsolute(relative)) {
    res.status(400).json({ error: 'invalid_asset' })
    return
  }

  try {
    await sendFileStream(req, res, assetPath)
  } catch {
    res.status(404).json({ error: 'asset_not_found' })
  }
})

const sendCorsHeaders = (res) => {
  res.setHeader('Access-Control-Allow-Origin', '*')
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type, Range')
  res.setHeader('Access-Control-Allow-Methods', 'GET, OPTIONS')
}

app.options('/proxy', (_req, res) => {
  sendCorsHeaders(res)
  res.status(204).end()
})

app.get('/proxy', async (req, res) => {
  const target = req.query.url
  if (typeof target !== 'string') {
    res.status(400).json({ error: 'missing_url' })
    return
  }

  const validation = validateProxyUrl(target)
  if (!validation.ok) {
    res.status(400).json({ error: 'invalid_url', reason: validation.reason })
    return
  }

  const controller = new AbortController()
  const timeoutId = setTimeout(() => controller.abort(), 15000)
  const rangeHeader = req.get('range')
  const userAgent = req.get('user-agent')
  const acceptHeader = req.get('accept')

  const requestHeaders = {}
  if (rangeHeader) {
    requestHeaders.Range = rangeHeader
  }
  if (userAgent) {
    requestHeaders['User-Agent'] = userAgent
  }
  if (acceptHeader) {
    requestHeaders.Accept = acceptHeader
  }

  try {
    const response = await fetch(target, {
      headers: requestHeaders,
      signal: controller.signal,
    })
    res.status(response.status)
    sendCorsHeaders(res)

    const contentType = response.headers.get('content-type') || ''
    const pathname = new URL(target).pathname
    const isPlaylist =
      contentType.includes('application/vnd.apple.mpegurl') ||
      contentType.includes('application/x-mpegURL') ||
      pathname.endsWith('.m3u8')

    if (isPlaylist) {
      const body = await response.text()
      const forwardedProto = req.get('x-forwarded-proto')
      const forwardedHost = req.get('x-forwarded-host')
      const protocol = forwardedProto ? forwardedProto.split(',')[0] : req.protocol
      const hostHeader = forwardedHost ? forwardedHost.split(',')[0] : req.get('host')
      const proxyBase = `${protocol}://${hostHeader}`
      res.setHeader('Content-Type', contentType || 'application/vnd.apple.mpegurl')
      res.send(rewritePlaylist(body, target, proxyBase))
      return
    }

    res.setHeader('Content-Type', contentType || 'application/octet-stream')

    if (response.body) {
      if (typeof response.body.pipe === 'function') {
        response.body.pipe(res)
      } else if (typeof Readable.fromWeb === 'function') {
        Readable.fromWeb(response.body).pipe(res)
      } else {
        const buffer = Buffer.from(await response.arrayBuffer())
        res.end(buffer)
      }
    } else {
      res.end()
    }
  } catch (error) {
    const status = error?.name === 'AbortError' ? 504 : 502
    res.status(status).json({ error: 'proxy_failed' })
  } finally {
    clearTimeout(timeoutId)
  }
})

const distDir = path.join(__dirname, '..', 'hls-player', 'dist')
app.use(express.static(distDir))
app.get('*', (_req, res, next) => {
  res.sendFile(path.join(distDir, 'index.html'), (err) => {
    if (err) {
      next()
    }
  })
})

app.listen(port, host, () => {
  // eslint-disable-next-line no-console
  console.log(`Backend server listening on http://${host}:${port}`)
})
