import fs from 'node:fs/promises'
import path from 'node:path'

const SUPPORTED_EXTENSIONS = new Set(['.mp4', '.webm', '.m3u8', '.ts'])

const MEDIA_TYPES = new Map([
  ['.m3u8', 'application/vnd.apple.mpegurl'],
  ['.ts', 'video/mp2t'],
  ['.m4s', 'video/iso.segment'],
  ['.mp4', 'video/mp4'],
  ['.webm', 'video/webm'],
  ['.aac', 'audio/aac'],
  ['.vtt', 'text/vtt'],
  ['.key', 'application/octet-stream'],
])

export const resolveMediaRoot = async (mediaRoot) => {
  if (!mediaRoot) {
    throw new Error('missing_media_root')
  }
  return fs.realpath(mediaRoot)
}

export const sanitizeRelativePath = (value) => {
  if (!value || typeof value !== 'string') {
    throw new Error('invalid_path')
  }
  const normalized = value.replace(/\\/g, '/').trim()
  if (!normalized || normalized.startsWith('/') || normalized.includes('\0')) {
    throw new Error('invalid_path')
  }
  if (path.isAbsolute(normalized)) {
    throw new Error('invalid_path')
  }
  if (normalized.split('/').includes('..')) {
    throw new Error('invalid_path')
  }
  return normalized
}

export const resolveRelativePath = async (rootReal, relativePath) => {
  const sanitized = sanitizeRelativePath(relativePath)
  const absolute = path.resolve(rootReal, sanitized)
  const real = await fs.realpath(absolute)
  const relative = path.relative(rootReal, real)
  if (relative.startsWith('..') || path.isAbsolute(relative)) {
    throw new Error('path_outside_root')
  }
  return { absolute: real, relative }
}

export const listEntries = async (rootReal, relativeDir) => {
  const relative = relativeDir ? sanitizeRelativePath(relativeDir) : ''
  const absolute = relative ? path.resolve(rootReal, relative) : rootReal
  const real = await fs.realpath(absolute)
  const base = path.relative(rootReal, real)
  if (base.startsWith('..') || path.isAbsolute(base)) {
    throw new Error('path_outside_root')
  }

  const entries = await fs.readdir(real, { withFileTypes: true })
  return entries
    .map((entry) => {
      const ext = path.extname(entry.name).toLowerCase()
      return {
        name: entry.name,
        type: entry.isDirectory() ? 'dir' : 'file',
        ext,
      }
    })
    .filter((entry) => entry.type === 'dir' || SUPPORTED_EXTENSIONS.has(entry.ext))
    .sort((a, b) => a.name.localeCompare(b.name))
}

export const detectKind = (absolutePath) => {
  const ext = path.extname(absolutePath).toLowerCase()
  if (ext === '.m3u8' || ext === '.ts') {
    return 'hls'
  }
  return 'file'
}

export const getMimeType = (absolutePath) => {
  const ext = path.extname(absolutePath).toLowerCase()
  return MEDIA_TYPES.get(ext) || 'application/octet-stream'
}

export const isHttpUrl = (value) => {
  try {
    const url = new URL(value)
    return url.protocol === 'http:' || url.protocol === 'https:'
  } catch {
    return false
  }
}

export const rewriteLocalPlaylist = ({
  playlistBody,
  playlistPath,
  shareId,
  rootReal,
  publicBase,
}) => {
  const baseDir = path.dirname(playlistPath)
  const assetBase = `${publicBase.replace(/\/$/, '')}/local/share/hls/${shareId}`

  const resolveAsset = (assetPath) => {
    if (isHttpUrl(assetPath)) {
      return assetPath
    }
    const cleaned = assetPath.split('?')[0].split('#')[0]
    const absolute = path.resolve(baseDir, cleaned)
    const relative = path.relative(rootReal, absolute)
    if (relative.startsWith('..') || path.isAbsolute(relative)) {
      throw new Error('path_outside_root')
    }
    const encoded = encodeURIComponent(relative)
    return `${assetBase}/asset?path=${encoded}`
  }

  return playlistBody
    .split(/\r?\n/)
    .map((line) => {
      if (!line) {
        return line
      }
      if (line.startsWith('#')) {
        if (!line.includes('URI="')) {
          return line
        }
        return line.replace(/URI="([^"]+)"/g, (_match, uri) => {
          const resolved = resolveAsset(uri)
          return `URI="${resolved}"`
        })
      }
      if (isHttpUrl(line)) {
        return line
      }
      return resolveAsset(line)
    })
    .join('\n')
}
