import fs from 'node:fs/promises'
import path from 'node:path'

const DEFAULT_STORE = { shares: {} }

const nowSeconds = () => Math.floor(Date.now() / 1000)

export const createShareStore = ({ filePath }) => {
  let cache = null
  let loadPromise = null
  let writeQueue = Promise.resolve()

  const ensureDir = async () => {
    await fs.mkdir(path.dirname(filePath), { recursive: true })
  }

  const load = async () => {
    if (cache) {
      return cache
    }
    if (!loadPromise) {
      loadPromise = (async () => {
        try {
          const raw = await fs.readFile(filePath, 'utf8')
          cache = JSON.parse(raw)
        } catch (error) {
          if (error.code !== 'ENOENT') {
            throw error
          }
          cache = { ...DEFAULT_STORE }
        }
        if (!cache || typeof cache !== 'object' || !cache.shares) {
          cache = { ...DEFAULT_STORE }
        }
        return cache
      })()
    }
    return loadPromise
  }

  const persist = async () => {
    await ensureDir()
    const payload = JSON.stringify(cache, null, 2)
    writeQueue = writeQueue.then(() => fs.writeFile(filePath, payload))
    await writeQueue
  }

  const pruneExpired = (store) => {
    const now = nowSeconds()
    let changed = false
    for (const [shareId, share] of Object.entries(store.shares)) {
      if (share.expiresAt && share.expiresAt <= now) {
        delete store.shares[shareId]
        changed = true
      }
    }
    return changed
  }

  const list = async () => {
    const store = await load()
    const changed = pruneExpired(store)
    if (changed) {
      await persist()
    }
    return Object.entries(store.shares).map(([shareId, share]) => ({
      shareId,
      ...share,
    }))
  }

  const get = async (shareId) => {
    const store = await load()
    const share = store.shares[shareId]
    if (!share) {
      return { status: 'missing' }
    }
    if (share.expiresAt && share.expiresAt <= nowSeconds()) {
      delete store.shares[shareId]
      await persist()
      return { status: 'expired' }
    }
    return { status: 'active', share: { shareId, ...share } }
  }

  const create = async ({ shareId, path: relPath, kind, ttlSeconds, posterUrl }) => {
    const store = await load()
    const createdAt = nowSeconds()
    const expiresAt = createdAt + ttlSeconds
    store.shares[shareId] = {
      path: relPath,
      kind,
      createdAt,
      expiresAt,
      posterUrl: posterUrl || null,
    }
    await persist()
    return { shareId, ...store.shares[shareId] }
  }

  const update = async (shareId, updates) => {
    const store = await load()
    const share = store.shares[shareId]
    if (!share) {
      return { status: 'missing' }
    }
    if (typeof updates.posterUrl !== 'undefined') {
      share.posterUrl = updates.posterUrl || null
    }
    await persist()
    return { status: 'updated', share: { shareId, ...share } }
  }

  const revoke = async (shareId) => {
    const store = await load()
    if (!store.shares[shareId]) {
      return false
    }
    delete store.shares[shareId]
    await persist()
    return true
  }

  return {
    list,
    get,
    create,
    update,
    revoke,
  }
}
