import express from 'express'
import fs from 'node:fs/promises'
import path from 'node:path'
import { fileURLToPath } from 'node:url'

import { buildCopyArgs, buildTranscodeArgs } from './hlsPackager.js'
import { resolveMediaRoot, resolveRelativePath } from './localMedia.js'

const loadEnvFile = async (envFilePath) => {
  if (!envFilePath) {
    return
  }
  try {
    const raw = await fs.readFile(envFilePath, 'utf8')
    raw.split(/\r?\n/).forEach((line) => {
      const trimmed = line.trim()
      if (!trimmed || trimmed.startsWith('#') || !trimmed.includes('=')) {
        return
      }
      const [key, ...rest] = trimmed.split('=')
      if (!(key in process.env)) {
        process.env[key] = rest.join('=').trim()
      }
    })
  } catch {
    // Ignore missing env file.
  }
}

const ensureDir = async (dir) => {
  await fs.mkdir(dir, { recursive: true })
}

const runCommand = async (command, args) => {
  const { spawn } = await import('node:child_process')
  return new Promise((resolve, reject) => {
    const child = spawn(command, args)
    let stderr = ''

    child.stderr.on('data', (chunk) => {
      stderr += chunk.toString()
    })

    child.on('error', (error) => reject(error))
    child.on('close', (code) => {
      if (code === 0) {
        resolve()
        return
      }
      const error = new Error(`ffmpeg_failed:${code}`)
      error.details = stderr
      reject(error)
    })
  })
}

const prepareOutputDir = async (dir) => {
  await fs.rm(dir, { recursive: true, force: true })
  await ensureDir(dir)
}

const parseBoolean = (value, fallback) => {
  if (typeof value === 'undefined') {
    return fallback
  }
  const normalized = String(value).trim().toLowerCase()
  if (['1', 'true', 'yes', 'on'].includes(normalized)) {
    return true
  }
  if (['0', 'false', 'no', 'off'].includes(normalized)) {
    return false
  }
  return fallback
}

const isAuthorized = (req, token) => {
  if (!token) {
    return false
  }
  const header = req.get('authorization') || ''
  return header === `Bearer ${token}`
}

const boot = async () => {
  const filename = fileURLToPath(import.meta.url)
  const rootDir = path.resolve(path.dirname(filename), '..')
  const envFile = process.env.PACKAGER_ENV_FILE || path.join(rootDir, 'hls-player', '.env')
  await loadEnvFile(envFile)

  const app = express()
  app.use(express.json({ limit: '2mb' }))

  const packagerToken = process.env.LOCAL_PACKAGER_TOKEN || ''
  const ffmpegPath = process.env.FFMPEG_PATH || 'ffmpeg'
  const transcodeTarget = process.env.LOCAL_TRANSCODE_TARGET || 'auto'
  // Default to using VideoToolbox (Apple's hardware-accelerated video API) on macOS.
  // This aligns with ffmpeg's support for VideoToolbox on darwin while keeping it
  // disabled by default on other platforms. The env var can always override this.
  const useVideoToolbox = parseBoolean(
    process.env.LOCAL_PACKAGER_USE_VIDEOTOOLBOX,
    process.platform === 'darwin',
  )
  const mediaRootHost = process.env.LOCAL_MEDIA_ROOT_HOST || process.env.LOCAL_MEDIA_ROOT || ''
  const dataRootHost = process.env.LOCAL_DATA_DIR_HOST || ''
  const host = process.env.LOCAL_PACKAGER_HOST || '127.0.0.1'
  const port = Number(process.env.LOCAL_PACKAGER_PORT || 7799)

  if (!packagerToken) {
    throw new Error('LOCAL_PACKAGER_TOKEN is required for packager-server')
  }

  if (!mediaRootHost || !dataRootHost) {
    throw new Error('LOCAL_MEDIA_ROOT_HOST and LOCAL_DATA_DIR_HOST are required')
  }

  const rootReal = await resolveMediaRoot(mediaRootHost)

  app.post('/packager/hlsify', async (req, res) => {
    if (!isAuthorized(req, packagerToken)) {
      res.status(401).json({ error: 'unauthorized' })
      return
    }

    const { shareId, relativePath, target } = req.body || {}
    if (!shareId || !relativePath) {
      res.status(400).json({ error: 'missing_params' })
      return
    }

    try {
      const resolved = await resolveRelativePath(rootReal, relativePath)
      const outputDir = path.join(dataRootHost, 'hls-cache', shareId)
      await prepareOutputDir(outputDir)

      try {
        await runCommand(ffmpegPath, buildCopyArgs({ input: resolved.absolute, outputDir }))
      } catch (error) {
        const mode = target || transcodeTarget
        if (mode === 'copy') {
          throw error
        }
        const codecTarget = mode === 'hevc' ? 'hevc' : 'h264'
        await runCommand(ffmpegPath, buildTranscodeArgs({
          input: resolved.absolute,
          outputDir,
          target: codecTarget,
          useVideoToolbox,
        }))
      }
      const detail =
        error && typeof error === 'object' && 'details' in error && error.details
          ? error.details
          : error && typeof error === 'object' && 'message' in error && error.message
            ? error.message
            : String(error)

      res.status(500).json({
        error: 'packager_failed',
        detail,
      res.status(500).json({
        error: 'packager_failed',
        detail: error.details || error.message,
      })
    }
  })

  app.listen(port, host, () => {
    // eslint-disable-next-line no-console
    console.log(`Packager server listening on http://${host}:${port}`)
  })
}

boot().catch((error) => {
  // eslint-disable-next-line no-console
  console.error('Packager server failed to start', error)
  process.exit(1)
})
