const PRIVATE_RANGES = [
  /^localhost$/i,
  /^127\./,
  /^10\./,
  /^192\.168\./,
  /^172\.(1[6-9]|2\d|3[0-1])\./,
]

const isPrivateHost = (host) => {
  if (!host) {
    return true
  }
  if (host === '::1') {
    return true
  }
  return PRIVATE_RANGES.some((pattern) => pattern.test(host))
}

export const validateProxyUrl = (rawUrl) => {
  try {
    const url = new URL(rawUrl)
    if (!['http:', 'https:'].includes(url.protocol)) {
      return { ok: false, reason: 'unsupported_protocol' }
    }
    if (isPrivateHost(url.hostname)) {
      return { ok: false, reason: 'private_host' }
    }
    return { ok: true }
  } catch (error) {
    return { ok: false, reason: 'invalid_url' }
  }
}


const buildProxyUrl = (proxyBase, targetUrl) => {
  const base = proxyBase.replace(/\/$/, '')
  const encoded = encodeURIComponent(targetUrl)
  return `${base}/proxy?url=${encoded}`
}

const isProxyUrl = (proxyBase, targetUrl) => {
  try {
    const url = new URL(targetUrl)
    const base = new URL(proxyBase)
    return url.origin === base.origin && url.pathname === '/proxy'
  } catch (error) {
    return false
  }
}

export const rewritePlaylist = (playlistBody, playlistUrl, proxyBase) => {
  const baseUrl = new URL(playlistUrl)

  return playlistBody
    .split(/\r?\n/)
    .map((line) => {
      if (!line) {
        return line
      }

      if (line.startsWith('#')) {
        if (!line.includes('URI="')) {
          return line
        }

        return line.replace(/URI="([^"]+)"/g, (_match, uri) => {
          const absolute = new URL(uri, baseUrl).toString()
          if (isProxyUrl(proxyBase, absolute)) {
            return `URI="${absolute}"`
          }
          return `URI="${buildProxyUrl(proxyBase, absolute)}"`
        })
      }

      const absolute = new URL(line, baseUrl).toString()
      if (isProxyUrl(proxyBase, absolute)) {
        return absolute
      }
      return buildProxyUrl(proxyBase, absolute)
    })
    .join('\n')
}
