import fs from 'node:fs/promises'
import os from 'node:os'
import path from 'node:path'
import test from 'node:test'
import assert from 'node:assert/strict'

import { createShareStore } from '../localMediaStore.js'
import { resolveMediaRoot, resolveRelativePath, rewriteLocalPlaylist } from '../localMedia.js'

test('resolveRelativePath blocks traversal outside the media root', async () => {
  // Purpose: prevent escaping the configured media root using path traversal.
  const rootDir = await fs.mkdtemp(path.join(os.tmpdir(), 'local-media-'))
  await fs.mkdir(path.join(rootDir, 'videos'))
  await fs.writeFile(path.join(rootDir, 'videos', 'sample.mp4'), 'data')
  const rootReal = await resolveMediaRoot(rootDir)

  const resolved = await resolveRelativePath(rootReal, 'videos/sample.mp4')
  assert.equal(resolved.relative, path.join('videos', 'sample.mp4'))

  await assert.rejects(() => resolveRelativePath(rootReal, '../outside.mp4'))
})

test('share store expires entries based on TTL', async () => {
  // Purpose: ensure expired shares are not treated as active playback tokens.
  const rootDir = await fs.mkdtemp(path.join(os.tmpdir(), 'local-shares-'))
  const dbPath = path.join(rootDir, 'shares.json')
  const store = createShareStore({ filePath: dbPath })

  await store.create({ shareId: 'share-1', path: 'video.mp4', kind: 'file', ttlSeconds: 1 })
  const originalNow = Date.now
  try {
    Date.now = () => originalNow() + 2000

    const result = await store.get('share-1')
    assert.equal(result.status, 'expired')
  } finally {
    Date.now = originalNow
  }
})

test('rewriteLocalPlaylist rewrites relative segment and key URIs', async () => {
  // Purpose: ensure local playlists use backend asset routes for segments and keys.
  const playlistBody = [
    '#EXTM3U',
    '#EXT-X-KEY:METHOD=AES-128,URI="keys/key.key"',
    '#EXTINF:4.0,',
    'segment-001.ts',
  ].join('\n')

  const rewritten = rewriteLocalPlaylist({
    playlistBody,
    playlistPath: '/media/streams/playlist.m3u8',
    shareId: 'share-9',
    rootReal: '/media',
    publicBase: 'http://localhost:2738',
  })

  assert.match(rewritten, /local\/share\/hls\/share-9\/asset\?path=streams%2Fkeys%2Fkey.key/)
  assert.match(rewritten, /local\/share\/hls\/share-9\/asset\?path=streams%2Fsegment-001.ts/)
})
