import test from 'node:test'
import assert from 'node:assert/strict'

import { validateProxyUrl, rewritePlaylist } from '../proxy.js'

test('rejects non-http(s) urls', () => {
  const result = validateProxyUrl('file:///etc/passwd')
  assert.equal(result.ok, false)
})

test('rejects localhost urls', () => {
  const result = validateProxyUrl('http://localhost:8080/stream.m3u8')
  assert.equal(result.ok, false)
})

test('rejects private ip urls', () => {
  const result = validateProxyUrl('http://10.0.0.1/stream.m3u8')
  assert.equal(result.ok, false)
})

test('accepts https public url', () => {
  const result = validateProxyUrl('https://example.com/stream.m3u8')
  assert.equal(result.ok, true)
})


test('rewrites playlist urls to proxy', () => {
  const playlist = [
    '#EXTM3U',
    '#EXT-X-KEY:METHOD=AES-128,URI="key.key"',
    '#EXTINF:6.0,',
    'segments/segment-1.ts',
  ].join('\n')

  const result = rewritePlaylist(
    playlist,
    'https://example.com/live/master.m3u8',
    'http://127.0.0.1:2738',
  )

  const keyUrl = encodeURIComponent('https://example.com/live/key.key')
  const segmentUrl = encodeURIComponent('https://example.com/live/segments/segment-1.ts')

  assert.ok(result.includes(`URI="http://127.0.0.1:2738/proxy?url=${keyUrl}"`))
  assert.ok(result.includes(`http://127.0.0.1:2738/proxy?url=${segmentUrl}`))
})

test('rewrites absolute playlist urls to proxy', () => {
  const playlist = [
    '#EXTM3U',
    'https://cdn.example.com/variant.m3u8',
  ].join('\n')

  const result = rewritePlaylist(
    playlist,
    'https://example.com/live/master.m3u8',
    'http://127.0.0.1:2738',
  )

  const encoded = encodeURIComponent('https://cdn.example.com/variant.m3u8')
  assert.ok(result.includes(`http://127.0.0.1:2738/proxy?url=${encoded}`))
})
