import { useEffect, useMemo, useRef, useState } from 'react'
import { createPortal } from 'react-dom'

import { encodeConfig } from '../lib/configCodec'
import type { PlayerConfig } from '../types/playerConfig'

interface EmbedSnippetProps {
  open: boolean
  config: PlayerConfig
  onClose: () => void
}

function EmbedSnippet({ open, config, onClose }: EmbedSnippetProps) {
  const [copied, setCopied] = useState(false)
  const [copyError, setCopyError] = useState<string | null>(null)
  const closeButtonRef = useRef<HTMLButtonElement | null>(null)

  useEffect(() => {
    if (!open) {
      setCopied(false)
      setCopyError(null)
      return
    }

    const onKeyDown = (event: KeyboardEvent) => {
      if (event.key === 'Escape') {
        onClose()
      }
    }

    window.addEventListener('keydown', onKeyDown)
    return () => window.removeEventListener('keydown', onKeyDown)
  }, [open, onClose])

  useEffect(() => {
    if (!open || typeof document === 'undefined') {
      return
    }
    const previousOverflow = document.body.style.overflow
    document.body.style.overflow = 'hidden'
    closeButtonRef.current?.focus()
    return () => {
      document.body.style.overflow = previousOverflow
    }
  }, [open])

  const { snippet, url } = useMemo(() => {
    const params = encodeConfig(config)
    const base = `${window.location.origin}/embed?${params.toString()}`
    const responsiveSnippet = [
      '<div style="position:relative;width:100%;padding-top:56.25%;">',
      `  <iframe src="${base}" style="position:absolute;top:0;left:0;width:100%;height:100%;" allow="autoplay; fullscreen; picture-in-picture" loading="lazy" frameborder="0"></iframe>`,
      '</div>',
    ].join('\n')

    return {
      url: base,
      snippet: responsiveSnippet,
    }
  }, [config])

  const onCopy = async () => {
    try {
      if (navigator?.clipboard?.writeText) {
        await navigator.clipboard.writeText(snippet)
      } else if (typeof document !== 'undefined') {
        const textarea = document.createElement('textarea')
        textarea.value = snippet
        textarea.setAttribute('readonly', 'true')
        textarea.style.position = 'absolute'
        textarea.style.left = '-9999px'
        document.body.appendChild(textarea)
        textarea.select()
        const success = document.execCommand('copy')
        document.body.removeChild(textarea)
        if (!success) {
          throw new Error('execCommand failed')
        }
      } else {
        throw new Error('clipboard unavailable')
      }
      setCopyError(null)
      setCopied(true)
      window.setTimeout(() => setCopied(false), 1600)
    } catch {
      setCopyError('Copy failed. Please select and copy manually.')
    }
  }

  if (!open) {
    return null
  }

  const modal = (
    <div className="modal-overlay" role="dialog" aria-modal="true" onClick={onClose}>
      <div className="modal-card" onClick={(event) => event.stopPropagation()}>
        <div className="modal-header">
          <div>
            <p className="kicker">Publish</p>
            <h2 className="modal-title">Embed this player</h2>
            <p className="workspace-subtitle">
              Copy the responsive iframe and paste it into any page.
            </p>
          </div>
          <button
            ref={closeButtonRef}
            type="button"
            onClick={onClose}
            className="modal-close"
          >
            Close
          </button>
        </div>

        <div className="modal-code">{snippet}</div>

        <div className="modal-meta">
          <span>Embed URL</span>
          <span>{url}</span>
        </div>

        <div className="modal-actions">
          <button type="button" onClick={onCopy} className="modal-copy">
            Copy Embed
          </button>
          {copied ? <span className="modal-copied">Copied</span> : null}
          {copyError ? <span className="modal-error">{copyError}</span> : null}
        </div>
      </div>
    </div>
  )

  if (typeof document === 'undefined') {
    return modal
  }

  return createPortal(modal, document.body)
}

export default EmbedSnippet
