import React from 'react'
import { render, screen } from '@testing-library/react'
import { beforeEach, describe, expect, it, vi } from 'vitest'

class PlyrMock {
  destroy = vi.fn()
  toggleControls = vi.fn()
  autoplay = false
  muted = false
  loop = { active: false }
  constructor() {}
}

vi.mock('plyr', () => ({
  default: PlyrMock,
}))

const attachHlsMock = vi.fn(() => () => {})

vi.mock('../lib/hlsController', () => ({
  attachHls: attachHlsMock,
}))

const baseConfig = {
  url: '',
  autoplay: false,
  muted: false,
  loop: false,
  poster: '',
  controls: true,
  preload: 'metadata' as const,
  proxy: false,
}

describe('HlsPlayer', () => {
  beforeEach(() => {
    attachHlsMock.mockClear()
  })

  it('routes local hls tokens through the HLS pipeline', async () => {
    // Purpose: ensure localhls tokens resolve to backend HLS URLs and attach via hls.js.
    const { default: HlsPlayer } = await import('./HlsPlayer')
    render(
      <HlsPlayer
        config={{ ...baseConfig, url: 'localhls:share-123' }}
      />
    )
    const video = await screen.findByTestId('hls-video')
    expect(attachHlsMock).toHaveBeenCalledWith(
      video,
      expect.stringContaining('/local/share/hls/share-123/index.m3u8'),
      expect.any(Function),
    )
  })

  it('uses native playback for local file tokens', async () => {
    // Purpose: localfile tokens should bypass hls.js and target the file endpoint.
    const { default: HlsPlayer } = await import('./HlsPlayer')
    render(
      <HlsPlayer
        config={{ ...baseConfig, url: 'localfile:share-456' }}
      />
    )
    const video = await screen.findByTestId('hls-video')
    const src = video.getAttribute('src') || video.src
    expect(attachHlsMock).not.toHaveBeenCalled()
    expect(src).toContain('/local/share/file/share-456')
  })

  it('renders a video element', async () => {
    // Purpose: ensure the video node is mounted for playback controls and styling.
    const { default: HlsPlayer } = await import('./HlsPlayer')
    render(
      <HlsPlayer
        config={{ ...baseConfig, url: 'https://example.com/stream.m3u8' }}
      />,
    )
    const video = await screen.findByTestId('hls-video')
    expect(video).toBeInTheDocument()
  })

  it('does not show a missing url banner when idle', async () => {
    // Purpose: avoid showing an error banner before a user has entered a URL.
    const { default: HlsPlayer } = await import('./HlsPlayer')
    render(<HlsPlayer config={baseConfig} />)
    expect(screen.queryByText(/missing stream url/i)).not.toBeInTheDocument()
  })
})
