import { useCallback, useEffect, useLayoutEffect, useMemo, useRef, useState } from 'react'
import Plyr from 'plyr'
import 'plyr/dist/plyr.css'

import type { PlayerConfig } from '../types/playerConfig'
import { resolveBackendUrl } from '../lib/backendUrl'
import { attachHls } from '../lib/hlsController'
import type { HlsStatus } from '../lib/hlsController'
import StatusBanner from './StatusBanner'

interface HlsPlayerProps {
  config: PlayerConfig
  onStatusChange?: (status: HlsStatus) => void
}

const statusCopy = (status: HlsStatus) => {
  if (status.state === 'loading') {
    return status.message || 'Loading stream…'
  }
  if (status.state === 'error') {
    return status.message
  }
  return ''
}

const resolveProxyUrl = (backendUrl: string, url: string) => {
  const encoded = encodeURIComponent(url)
  return `${backendUrl.replace(/\/$/, '')}/proxy?url=${encoded}`
}

const parseLocalToken = (value: string) => {
  if (!value) {
    return null
  }
  const [prefix, shareId] = value.split(':')
  if (!shareId) {
    return null
  }
  if (prefix !== 'localhls' && prefix !== 'localfile') {
    return null
  }
  return { kind: prefix, shareId }
}

const isHlsUrl = (value: string) => /\.m3u8($|\?)/i.test(value)

const safeDestroy = (instance: Plyr | null, video: HTMLVideoElement) => {
  if (!instance) {
    return
  }
  if (!video.isConnected) {
    return
  }
  try {
    instance.destroy()
  } catch {
    // Plyr can throw if React already removed nodes in StrictMode re-mounts.
  }
}

function HlsPlayer({ config, onStatusChange }: HlsPlayerProps) {
  const containerRef = useRef<HTMLDivElement | null>(null)
  const videoRef = useRef<HTMLVideoElement | null>(null)
  const playerRef = useRef<Plyr | null>(null)
  const [status, setStatus] = useState<HlsStatus>({ state: 'idle' })
  const [startedPlaybackKey, setStartedPlaybackKey] = useState<string | null>(null)

  const reportStatus = useCallback((next: HlsStatus) => {
    setStatus(next)
    onStatusChange?.(next)
  }, [onStatusChange])

  const backendUrl = resolveBackendUrl()
  const posterUrl = config.poster.trim()

  const playback = useMemo(() => {
    if (!config.url) {
      return { url: '', mode: 'hls' as const }
    }
    const localToken = parseLocalToken(config.url)
    if (localToken) {
      const base = backendUrl.replace(/\/$/, '')
      if (localToken.kind === 'localhls') {
        return { url: `${base}/local/share/hls/${localToken.shareId}/index.m3u8`, mode: 'hls' as const }
      }
      return { url: `${base}/local/share/file/${localToken.shareId}`, mode: 'native' as const }
    }
    const url = config.proxy ? resolveProxyUrl(backendUrl, config.url) : config.url
    return { url, mode: isHlsUrl(url) ? 'hls' as const : 'native' as const }
  }, [backendUrl, config.proxy, config.url])

  const playbackKey = `${playback.url}|${posterUrl}`
  const hasStartedPlayback = startedPlaybackKey === playbackKey

  useLayoutEffect(() => {
    const container = containerRef.current
    if (!container || videoRef.current) {
      return
    }

    const video = document.createElement('video')
    video.className = 'w-full h-full object-contain'
    video.dataset.testid = 'hls-video'
    video.playsInline = true
    container.appendChild(video)
    videoRef.current = video

    return () => {
      if (videoRef.current) {
        safeDestroy(playerRef.current, videoRef.current)
        if (videoRef.current.parentNode) {
          videoRef.current.parentNode.removeChild(videoRef.current)
        }
      }
      playerRef.current = null
      videoRef.current = null
    }
  }, [])

  useEffect(() => {
    const video = videoRef.current
    if (!video) {
      return
    }

    const handlePlaying = () => setStartedPlaybackKey(playbackKey)

    video.addEventListener('playing', handlePlaying)

    return () => {
      video.removeEventListener('playing', handlePlaying)
    }
  }, [playbackKey])

  useEffect(() => {
    const video = videoRef.current
    if (!video) {
      return
    }

    if (posterUrl && !hasStartedPlayback) {
      video.setAttribute('poster', posterUrl)
      video.setAttribute('data-poster', posterUrl)
    } else {
      video.removeAttribute('poster')
      video.removeAttribute('data-poster')
    }
  }, [hasStartedPlayback, posterUrl])

  useEffect(() => {
    const video = videoRef.current
    if (!video) {
      return
    }

    if (playerRef.current) {
      safeDestroy(playerRef.current, video)
      playerRef.current = null
    }

    const plyr = new Plyr(video, {
      autoplay: config.autoplay,
      muted: config.muted,
      loop: { active: config.loop },
      iconUrl: '/plyr.svg',
      controls: config.controls
        ? ['play-large', 'play', 'live', 'mute', 'volume', 'captions', 'settings', 'pip', 'airplay', 'fullscreen']
        : [],
      hideControls: false,
    })

    playerRef.current = plyr

    return () => {
      safeDestroy(plyr, video)
      if (playerRef.current === plyr) {
        playerRef.current = null
      }
    }
  }, [config.autoplay, config.controls, config.loop, config.muted])

  useEffect(() => {
    const video = videoRef.current
    if (!video) {
      return
    }

    if (!playback.url) {
      return
    }

    let active = true
    const pushStatus = (nextStatus: HlsStatus) => {
      queueMicrotask(() => {
        if (active) {
          reportStatus(nextStatus)
        }
      })
    }

    let detach = () => {}
    if (playback.mode === 'hls') {
      video.removeAttribute('src')
      video.load()
      detach = attachHls(video, playback.url, reportStatus)
    } else {
      pushStatus({ state: 'loading' })
      video.src = playback.url
      const handleReady = () => reportStatus({ state: 'ready' })
      const handleError = () => reportStatus({ state: 'error', message: 'Failed to load media.' })
      video.addEventListener('canplay', handleReady)
      video.addEventListener('error', handleError)
      video.load()
      detach = () => {
        video.removeEventListener('canplay', handleReady)
        video.removeEventListener('error', handleError)
        video.removeAttribute('src')
        video.load()
      }
    }

    video.preload = config.preload

    return () => {
      active = false
      detach()
    }
  }, [config.preload, playback, reportStatus])

  const showPoster = Boolean(posterUrl) && !hasStartedPlayback

  const effectiveStatus: HlsStatus = playback.url ? status : { state: 'idle' }
  const bannerMessage = statusCopy(effectiveStatus)
  const isError = effectiveStatus.state === 'error'

  return (
    <div
      className="relative w-full h-full flex items-center justify-center bg-black rounded-lg overflow-hidden shadow-2xl ring-1 ring-void-border"
      data-show-poster={showPoster ? 'true' : 'false'}
    >
      {bannerMessage ? (
        <div className="absolute top-4 left-0 right-0 z-50 flex justify-center pointer-events-none">
          <StatusBanner message={bannerMessage} tone={isError ? 'error' : 'info'} />
        </div>
      ) : null}
      <div ref={containerRef} className="w-full h-full relative flex items-center justify-center bg-black">
        {showPoster ? (
          <div className="player-poster" style={{ backgroundImage: `url("${posterUrl}")` }}>
            {/* Poster handled via CSS class in index.css or simple div here */}
            <div className="absolute inset-0 bg-cover bg-center bg-no-repeat" style={{ backgroundImage: `url("${posterUrl}")` }}>
              <div className="player-poster-control" aria-hidden="true" />
            </div>
          </div>
        ) : null}
        {/** Plyr will inject the video here */}
      </div>
    </div>
  )
}

export default HlsPlayer
