import { useEffect, useMemo, useState, type CSSProperties } from 'react'

import type { PlayerConfig } from '../types/playerConfig'
import type { HlsStatus } from '../lib/hlsController'
import { useDebounce } from '../hooks/useDebounce'
import { useUrlDraft } from '../hooks/useUrlDraft'
import { isGoogleVideoUrl } from '../lib/urlRules'

interface InputWorkspaceProps {
  config: PlayerConfig
  status?: HlsStatus
  onChange: (config: PlayerConfig) => void
  onPublish?: () => void
  showPublish?: boolean
}

const statusLabel = (status: HlsStatus, proxy: boolean) => {
  if (status.state === 'error') {
    return 'Error'
  }
  if (proxy && status.state !== 'idle') {
    return 'Proxy Active'
  }
  if (status.state === 'loading') {
    return 'Connecting'
  }
  if (status.state === 'ready') {
    return 'Ready'
  }
  return proxy ? 'Proxy Ready' : 'Ready'
}

const statusState = (status: HlsStatus, proxy: boolean) => {
  if (status.state === 'error') {
    return 'error'
  }
  if (proxy && status.state !== 'idle') {
    return 'active'
  }
  if (status.state === 'loading') {
    return 'loading'
  }
  return 'idle'
}

function InputWorkspace({
  config,
  status = { state: 'idle' },
  onChange,
  onPublish,
  showPublish = true,
}: InputWorkspaceProps) {
  const [advancedOpen, setAdvancedOpen] = useState(false)
  const [urlDraft, setUrlDraft] = useUrlDraft(config.url)
  const debouncedUrl = useDebounce(urlDraft, 520)

  useEffect(() => {
    if (debouncedUrl !== config.url) {
      onChange({
        ...config,
        url: debouncedUrl,
      })
    }
  }, [debouncedUrl, config, onChange])

  const badgeLabel = useMemo(
    () => statusLabel(status, config.proxy),
    [status, config.proxy],
  )

  const badgeState = useMemo(
    () => statusState(status, config.proxy),
    [status, config.proxy],
  )

  const isGoogleVideo = useMemo(() => isGoogleVideoUrl(config.url), [config.url])
  const showProxyAction =
    status.state === 'error' &&
    status.action === 'enable-proxy' &&
    !config.proxy &&
    !isGoogleVideo
  const canPublish = Boolean(config.url)

  return (
    <div className="workspace">
      <div className="workspace-header reveal" style={{ '--delay': '0.05s' } as CSSProperties}>
        <p className="kicker">Studio</p>
        <h1 className="workspace-title">High-Contrast HLS</h1>
        <p className="workspace-subtitle">
          Configure a single stream, refine playback, and publish a clean embed without
          the dashboard noise.
        </p>
      </div>

      <div className="reveal" style={{ '--delay': '0.12s' } as CSSProperties}>
        <label className="kicker" htmlFor="m3u8-url">
          Stream URL
        </label>
        <div className="smart-field">
          <input
            id="m3u8-url"
            aria-label="M3U8 URL"
            value={urlDraft}
            onChange={(event) => setUrlDraft(event.target.value)}
            placeholder="https://example.com/stream.m3u8"
            className="smart-field-input"
          />
          <span className="smart-field-badge" data-state={badgeState} title={badgeLabel}>
            B
          </span>
        </div>
        <div className="smart-field-meta">
          <span>{config.proxy ? 'Proxy routing on' : 'Direct origin'}</span>
          <div className="smart-field-actions">
            <span className={badgeState === 'error' ? 'status-error' : ''}>{badgeLabel}</span>
            {showProxyAction ? (
              <button
                type="button"
                className="inline-action"
                onClick={() => onChange({ ...config, proxy: true })}
              >
                Enable Proxy
              </button>
            ) : null}
          </div>
        </div>
      </div>

      <div className="reveal" style={{ '--delay': '0.2s' } as CSSProperties}>
        <p className="kicker">Action Sets</p>
        <div className="action-sets">
          <button
            type="button"
            className="action-button"
            aria-pressed={config.autoplay}
            onClick={() => onChange({ ...config, autoplay: !config.autoplay })}
          >
            <svg
              className="action-button__icon"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="1.6"
            >
              <path d="M7 5l10 7-10 7V5z" />
            </svg>
            Autoplay
            <span className="action-button__dot" />
          </button>
          <button
            type="button"
            className="action-button"
            aria-pressed={config.loop}
            onClick={() => onChange({ ...config, loop: !config.loop })}
          >
            <svg
              className="action-button__icon"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="1.6"
            >
              <path d="M3 12a6 6 0 016-6h7" />
              <path d="M14 3l3 3-3 3" />
              <path d="M21 12a6 6 0 01-6 6H8" />
              <path d="M10 21l-3-3 3-3" />
            </svg>
            Loop
            <span className="action-button__dot" />
          </button>
          <button
            type="button"
            className="action-button"
            aria-pressed={config.muted}
            onClick={() => onChange({ ...config, muted: !config.muted })}
          >
            <svg
              className="action-button__icon"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="1.6"
            >
              <path d="M5 9H3v6h2l5 4V5L5 9z" />
              <path d="M16 9l5 6" />
              <path d="M21 9l-5 6" />
            </svg>
            Muted
            <span className="action-button__dot" />
          </button>
        </div>
      </div>

      <div className="reveal" style={{ '--delay': '0.28s' } as CSSProperties}>
        <button
          type="button"
          className="refine-toggle"
          onClick={() => setAdvancedOpen((prev) => !prev)}
          aria-expanded={advancedOpen}
        >
          Refine
          <span>{advancedOpen ? '−' : '+'}</span>
        </button>
        {advancedOpen ? (
          <div className="refine-panel">
            <label className="switch">
              <span>CORS Proxy</span>
              <span className="switch-control">
                <input
                  type="checkbox"
                  className="switch-input"
                  aria-label="Use CORS Proxy"
                  checked={config.proxy}
                  onChange={(event) =>
                    onChange({
                      ...config,
                      proxy: event.target.checked,
                    })
                  }
                />
                <span className="switch-track" />
              </span>
            </label>

            <div className="field-grid">
              <label>
                <span className="field-label">Poster URL</span>
                <input
                  value={config.poster}
                  onChange={(event) =>
                    onChange({
                      ...config,
                      poster: event.target.value,
                    })
                  }
                  placeholder="https://example.com/poster.jpg"
                  className="field-input"
                />
              </label>
              <label>
                <span className="field-label">Preload</span>
                <select
                  value={config.preload}
                  onChange={(event) =>
                    onChange({
                      ...config,
                      preload: event.target.value as PlayerConfig['preload'],
                    })
                  }
                  className="field-select"
                >
                  <option value="metadata">Metadata</option>
                  <option value="auto">Auto</option>
                  <option value="none">None</option>
                </select>
              </label>
            </div>

            <label className="switch">
              <span>Controls</span>
              <span className="switch-control">
                <input
                  type="checkbox"
                  className="switch-input"
                  aria-label="Controls"
                  checked={config.controls}
                  onChange={(event) =>
                    onChange({
                      ...config,
                      controls: event.target.checked,
                    })
                  }
                />
                <span className="switch-track" />
              </span>
            </label>
          </div>
        ) : null}
      </div>

      {showPublish ? (
        <button
          type="button"
          onClick={() => onPublish?.()}
          disabled={!canPublish}
          aria-disabled={!canPublish}
          className="publish-button reveal"
          style={{ '--delay': '0.36s' } as CSSProperties}
        >
          {canPublish ? 'Copy Embed' : 'Add URL to publish'}
        </button>
      ) : null}
    </div>
  )
}

export default InputWorkspace
