
interface BrowseEntry {
  name: string
  type: 'dir' | 'file'
  ext?: string
}

interface FileBrowserDialogProps {
  isOpen: boolean
  dir: string
  entries: BrowseEntry[]
  loading: boolean
  error: string | null
  onClose: () => void
  onNavigate: (path: string) => void
  onSelect: (entry: BrowseEntry) => void
}

export function FileBrowserDialog({
  isOpen,
  dir,
  entries,
  loading,
  error,
  onClose,
  onNavigate,
  onSelect,
}: FileBrowserDialogProps) {
  if (!isOpen) return null

  const handleUp = () => {
    const segments = dir.split('/')
    segments.pop()
    onNavigate(segments.join('/'))
  }

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/80 backdrop-blur-sm p-4">
      <div className="bg-zinc-900 border border-zinc-800 rounded-xl shadow-2xl w-full max-w-2xl flex flex-col max-h-[80vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-4 border-b border-zinc-800 bg-zinc-900/50">
          <h3 className="text-zinc-100 font-medium text-sm tracking-wide">BROWSE LOCAL MEDIA</h3>
          <button 
            onClick={onClose}
            className="text-zinc-400 hover:text-white transition-colors"
          >
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M18 6 6 18"/><path d="m6 6 12 12"/></svg>
          </button>
        </div>

        {/* Path Bar */}
        <div className="flex items-center gap-2 p-3 bg-zinc-950/50 border-b border-zinc-800 text-xs font-mono text-zinc-400 overflow-x-auto">
          <span className="shrink-0 text-blue-500">/</span>
          {dir || <span className="opacity-50">root</span>}
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-2">
          {loading && (
            <div className="flex items-center justify-center py-8 text-zinc-500 text-sm">
              Loading...
            </div>
          )}
          
          {error && (
            <div className="p-4 rounded bg-red-900/20 border border-red-900/50 text-red-400 text-sm text-center">
              {error}
            </div>
          )}

          {!loading && !error && (
            <div className="grid gap-1">
              {dir && (
                <button
                  onClick={handleUp}
                  className="flex items-center gap-3 w-full p-2 rounded hover:bg-zinc-800 text-left group"
                >
                  <span className="text-blue-500 group-hover:text-blue-400">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="m15 18-6-6 6-6"/></svg>
                  </span>
                  <span className="text-zinc-400 font-mono text-sm">..</span>
                </button>
              )}
              
              {entries.map((entry) => (
                <button
                  key={`${entry.type}-${entry.name}`}
                  onClick={() => entry.type === 'dir' ? onNavigate(dir ? `${dir}/${entry.name}` : entry.name) : onSelect(entry)}
                  className="flex items-center gap-3 w-full p-2 rounded hover:bg-zinc-800 text-left group transition-colors"
                >
                  <span className={`shrink-0 ${entry.type === 'dir' ? 'text-blue-500' : 'text-zinc-500 group-hover:text-zinc-300'}`}>
                    {entry.type === 'dir' ? (
                       <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M20 20a2 2 0 0 0 2-2V8a2 2 0 0 0-2-2h-7.9a2 2 0 0 1-1.69-.9L9.6 3.9A2 2 0 0 0 7.93 3H4a2 2 0 0 0-2 2v13a2 2 0 0 0 2 2Z"/></svg>
                    ) : (
                       <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M15 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7Z"/><path d="M14 2v4a2 2 0 0 0 2 2h4"/></svg>
                    )}
                  </span>
                  <span className="text-zinc-300 font-mono text-sm truncate flex-1">{entry.name}</span>
                </button>
              ))}
              
              {entries.length === 0 && !dir && (
                <div className="text-center py-8 text-zinc-600 text-sm">
                  Root directory is empty
                </div>
              )}
              
              {entries.length === 0 && dir && (
                <div className="text-center py-8 text-zinc-600 text-sm">
                  Empty directory
                </div>
              )}
            </div>
          )}
        </div>
      </div>
    </div>
  )
}
