import { useMemo, useState } from 'react'

type UploadEntry = {
  name: string
  url: string
  updatedAt: number
}

type UploadedPostersDialogProps = {
  isOpen: boolean
  entries: UploadEntry[]
  loading: boolean
  error: string | null
  onClose: () => void
  onSelect: (entry: UploadEntry) => void
  onRefresh: () => void
}

export function UploadedPostersDialog({
  isOpen,
  entries,
  loading,
  error,
  onClose,
  onSelect,
  onRefresh,
}: UploadedPostersDialogProps) {
  const [selected, setSelected] = useState<UploadEntry | null>(null)

  const activeSelection = useMemo(() => {
    if (!isOpen || entries.length === 0) return null
    if (selected && entries.some((entry) => entry.url === selected.url)) {
      return selected
    }
    return entries[0]
  }, [entries, isOpen, selected])

  if (!isOpen) return null

  return (
    <div
      className="fixed inset-0 z-50 flex items-center justify-center bg-transparent backdrop-blur-md p-4"
      onClick={onClose}
      role="presentation"
    >
      <div
        className="bg-vn-red border border-vn-yellow/70 rounded-2xl shadow-2xl w-full max-w-4xl max-h-[85vh] flex flex-col overflow-hidden isolate"
        onClick={(event) => event.stopPropagation()}
        role="dialog"
        aria-modal="true"
      >
        <div className="flex items-center justify-between p-4 border-b border-vn-yellow/40 bg-vn-red">
          <div>
            <div className="text-xs font-bold tracking-[0.2em] text-vn-yellow">UPLOAD LIBRARY</div>
            <div className="text-[10px] text-vn-yellow/70">Select a poster to reuse</div>
          </div>
          <div className="flex items-center gap-2">
            <button
              onClick={onRefresh}
              className="px-3 py-1.5 text-[10px] font-bold rounded border border-vn-yellow/60 text-vn-yellow hover:bg-vn-yellow/10"
            >
              REFRESH
            </button>
            <button
              onClick={onClose}
              className="px-2 py-1 text-vn-yellow hover:bg-vn-yellow/10 rounded"
              aria-label="Close"
            >
              ✕
            </button>
          </div>
        </div>

        <div className="flex-1 overflow-y-auto p-4 bg-vn-red">
          {loading && <div className="text-vn-yellow/70 text-sm">LOADING...</div>}
          {error && <div className="text-vn-yellow text-sm">{error}</div>}
          {!loading && !error && entries.length === 0 && (
            <div className="text-vn-yellow/70 text-sm">No uploads found.</div>
          )}
          {!loading && !error && entries.length > 0 && (
            <div className="flex flex-col lg:flex-row gap-4">
              <div className="lg:w-1/2">
                <div className="grid grid-cols-3 gap-2">
                  {entries.map((entry) => (
                    <button
                      key={entry.url}
                      onClick={() => setSelected(entry)}
                      className={`rounded border ${activeSelection?.url === entry.url ? 'border-vn-yellow' : 'border-vn-yellow/40'} p-1 hover:border-vn-yellow`}
                      title={entry.name}
                    >
                      <img
                        src={entry.url}
                        alt={entry.name}
                        className="h-20 w-full object-cover rounded"
                        loading="lazy"
                      />
                    </button>
                  ))}
                </div>
              </div>
              <div className="lg:w-1/2 flex flex-col gap-3">
                <div className="text-[11px] text-vn-yellow/80 font-mono truncate">
                  {activeSelection?.name || 'Select an image'}
                </div>
                <div className="flex-1 border border-vn-yellow/40 rounded p-2">
                  {activeSelection ? (
                    <img
                      src={activeSelection.url}
                      alt={activeSelection.name}
                      className="w-full max-h-[360px] object-contain"
                    />
                  ) : (
                    <div className="text-vn-yellow/60 text-sm">Preview will appear here.</div>
                  )}
                </div>
                <button
                  onClick={() => activeSelection && onSelect(activeSelection)}
                  disabled={!activeSelection}
                  className="py-2 bg-vn-yellow text-vn-red font-bold text-xs rounded disabled:opacity-60"
                >
                  USE SELECTED POSTER
                </button>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  )
}
