const DEFAULT_BACKEND_URL = 'http://127.0.0.1:2738'
const PLACEHOLDER_HOST = 'your-domain.example'

type RuntimeEnv = {
  VITE_BACKEND_URL?: string
  BACKEND_PUBLIC_URL?: string
  BACKEND_URL?: string
}

const readRuntimeEnv = (): RuntimeEnv | undefined => {
  if (typeof window === 'undefined') {
    return undefined
  }
  return window.__HLS_PLAYER_ENV__
}

const readCurrentHostname = () => {
  if (typeof window === 'undefined') {
    return undefined
  }
  return window.location?.hostname
}

const readCurrentProtocol = () => {
  if (typeof window === 'undefined') {
    return undefined
  }
  return window.location?.protocol
}

const isLoopbackHost = (value: string | undefined) => (
  value === 'localhost' || value === '127.0.0.1' || value === '::1'
)

const normalizeCandidate = (value: string | undefined) => {
  const normalized = value?.trim()
  if (!normalized) {
    return undefined
  }
  if (normalized.includes(PLACEHOLDER_HOST)) {
    return undefined
  }
  return normalized
}

const parseCandidates = (value: string | undefined) => {
  const normalized = normalizeCandidate(value)
  if (!normalized) {
    return []
  }
  return normalized
    .split(',')
    .map((item) => item.trim())
    .filter((item) => item && !item.includes(PLACEHOLDER_HOST))
}

export const resolveBackendUrl = () => {
  const runtime = readRuntimeEnv()
  const candidates = [
    ...parseCandidates(runtime?.VITE_BACKEND_URL),
    ...parseCandidates(runtime?.BACKEND_PUBLIC_URL),
    ...parseCandidates(runtime?.BACKEND_URL),
    ...parseCandidates(import.meta.env.VITE_BACKEND_URL as string | undefined),
  ]

  const currentHostname = readCurrentHostname()
  if (currentHostname) {
    const match = candidates.find((candidate) => {
      try {
        return new URL(candidate).hostname === currentHostname
      } catch {
        return false
      }
    })
    if (match) {
      return match
    }
    const loopback = candidates.find((candidate) => {
      try {
        return isLoopbackHost(new URL(candidate).hostname)
      } catch {
        return false
      }
    })
    if (loopback && isLoopbackHost(currentHostname)) {
      const parsed = new URL(loopback)
      const protocol = parsed.protocol || readCurrentProtocol() || 'http:'
      const port = parsed.port || '2738'
      return `${protocol}//${currentHostname}:${port}`
    }
  }

  const first = candidates.find(Boolean)
  if (first) {
    return first
  }

  if (currentHostname) {
    const protocol = readCurrentProtocol() || 'http:'
    return `${protocol}//${currentHostname}:2738`
  }

  return DEFAULT_BACKEND_URL
}
