import { describe, expect, it } from 'vitest'
import { DEFAULT_PLAYER_CONFIG, decodeConfig, encodeConfig } from './configCodec'

describe('configCodec', () => {
  it('returns default config when params are empty', () => {
    const params = new URLSearchParams()
    expect(decodeConfig(params)).toEqual(DEFAULT_PLAYER_CONFIG)
  })

  it('encodes and decodes full config', () => {
    const config = {
      ...DEFAULT_PLAYER_CONFIG,
      url: 'https://example.com/stream.m3u8?token=abc',
      autoplay: true,
      muted: true,
      loop: true,
      controls: false,
      poster: 'https://example.com/poster.jpg',
      preload: 'auto' as const,
      proxy: true,
    }

    const params = encodeConfig(config)
    expect(params.get('src')).toBe(config.url)
    expect(params.get('auto')).toBe('1')
    expect(params.get('controls')).toBe('0')
    expect(params.get('proxy')).toBe('1')

    const decoded = decodeConfig(params)
    expect(decoded).toEqual(config)
  })

  it('accepts src + numeric flags', () => {
    const params = new URLSearchParams(
      'src=https%3A%2F%2Fexample.com%2Fstream.m3u8&auto=1&muted=1&controls=0&loop=1&proxy=1',
    )
    const decoded = decodeConfig(params)

    expect(decoded.url).toBe('https://example.com/stream.m3u8')
    expect(decoded.autoplay).toBe(true)
    expect(decoded.muted).toBe(true)
    expect(decoded.controls).toBe(false)
    expect(decoded.loop).toBe(true)
    expect(decoded.proxy).toBe(true)
  })

  it('falls back to defaults on invalid values', () => {
    const params = new URLSearchParams(
      'autoplay=maybe&controls=nope&preload=banana&muted=false&proxy=what',
    )
    const decoded = decodeConfig(params)

    expect(decoded.autoplay).toBe(false)
    expect(decoded.controls).toBe(true)
    expect(decoded.preload).toBe('metadata')
    expect(decoded.muted).toBe(false)
    expect(decoded.proxy).toBe(false)
  })
})
