import {
  DEFAULT_PLAYER_CONFIG,
  type PlayerConfig,
  type PreloadOption,
} from '../types/playerConfig'

const PRELOAD_OPTIONS: PreloadOption[] = ['auto', 'metadata', 'none']

const parseBoolean = (value: string | null, fallback: boolean) => {
  if (value === 'true' || value === '1') {
    return true
  }
  if (value === 'false' || value === '0') {
    return false
  }
  return fallback
}

const parsePreload = (value: string | null, fallback: PreloadOption) => {
  if (value && PRELOAD_OPTIONS.includes(value as PreloadOption)) {
    return value as PreloadOption
  }
  return fallback
}

const getUrlParam = (params: URLSearchParams) => {
  const url = params.get('url')?.trim()
  if (url) {
    return url
  }
  return params.get('src')?.trim() ?? ''
}

const getAutoplayParam = (params: URLSearchParams, fallback: boolean) => {
  if (params.has('autoplay')) {
    return parseBoolean(params.get('autoplay'), fallback)
  }
  return parseBoolean(params.get('auto'), fallback)
}

export const decodeConfig = (params: URLSearchParams): PlayerConfig => {
  return {
    ...DEFAULT_PLAYER_CONFIG,
    url: getUrlParam(params),
    autoplay: getAutoplayParam(params, DEFAULT_PLAYER_CONFIG.autoplay),
    muted: parseBoolean(params.get('muted'), DEFAULT_PLAYER_CONFIG.muted),
    loop: parseBoolean(params.get('loop'), DEFAULT_PLAYER_CONFIG.loop),
    poster: params.get('poster') ?? DEFAULT_PLAYER_CONFIG.poster,
    controls: parseBoolean(params.get('controls'), DEFAULT_PLAYER_CONFIG.controls),
    preload: parsePreload(params.get('preload'), DEFAULT_PLAYER_CONFIG.preload),
    proxy: parseBoolean(params.get('proxy'), DEFAULT_PLAYER_CONFIG.proxy),
  }
}

const setFlag = (params: URLSearchParams, key: string, value: boolean) => {
  params.set(key, value ? '1' : '0')
}

export const encodeConfig = (config: PlayerConfig): URLSearchParams => {
  const params = new URLSearchParams()

  params.set('src', config.url)
  setFlag(params, 'auto', config.autoplay)
  setFlag(params, 'muted', config.muted)
  setFlag(params, 'loop', config.loop)
  setFlag(params, 'controls', config.controls)
  setFlag(params, 'proxy', config.proxy)
  params.set('preload', config.preload)

  if (config.poster) {
    params.set('poster', config.poster)
  }

  return params
}

export { DEFAULT_PLAYER_CONFIG }
