import Hls from 'hls.js'

export type HlsStatus =
  | { state: 'idle' }
  | { state: 'loading'; message?: string }
  | { state: 'ready' }
  | { state: 'error'; message: string; action?: 'enable-proxy' }

const DEFAULT_MAX_RETRIES = 3

const errorMessage = (detail?: string, code?: number, isProxy?: boolean) => {
  if (code) {
    const label = isProxy ? 'Proxy' : 'Origin'
    return `${label} responded ${code}.`
  }
  return detail || 'Playback error'
}

export const attachHls = (
  video: HTMLVideoElement,
  url: string,
  onStatus: (status: HlsStatus) => void,
) => {
  let destroyed = false
  let retryCount = 0

  const report = (status: HlsStatus) => {
    if (!destroyed) {
      onStatus(status)
    }
  }

  if (Hls.isSupported()) {
    const hls = new Hls({
      autoStartLoad: true,
      maxBufferLength: 30,
      maxMaxBufferLength: 60,
    })

    report({ state: 'loading' })
    hls.loadSource(url)
    hls.attachMedia(video)

    hls.on(Hls.Events.MANIFEST_PARSED, () => {
      report({ state: 'ready' })
    })

    hls.on(Hls.Events.ERROR, (_event, data) => {
      if (destroyed) {
        return
      }

      if (data.fatal) {
        const responseCode = data?.response?.code
        const isProxyRequest = url.includes('/proxy?')
        const isCorsBlocked =
          data.type === Hls.ErrorTypes.NETWORK_ERROR &&
          data?.response?.code === 0

        if (isCorsBlocked) {
          if (isProxyRequest) {
            report({
              state: 'error',
              message: 'Proxy unavailable. Check backend connection.',
            })
          } else {
            report({
              state: 'error',
              message: 'CORS blocked. Enable proxy to continue.',
              action: 'enable-proxy',
            })
          }
          hls.destroy()
          return
        }

        if (retryCount < DEFAULT_MAX_RETRIES) {
          retryCount += 1
          if (data.type === Hls.ErrorTypes.NETWORK_ERROR) {
            report({
              state: 'loading',
              message: `Network error. Retrying (${retryCount}/${DEFAULT_MAX_RETRIES})…`,
            })
            hls.startLoad()
            return
          }
          if (data.type === Hls.ErrorTypes.MEDIA_ERROR) {
            report({
              state: 'loading',
              message: `Media error. Recovering (${retryCount}/${DEFAULT_MAX_RETRIES})…`,
            })
            hls.recoverMediaError()
            return
          }
        }

        report({
          state: 'error',
          message: errorMessage(data.details, responseCode, isProxyRequest),
        })
        hls.destroy()
      }
    })

    return () => {
      destroyed = true
      hls.destroy()
    }
  }

  if (video.canPlayType('application/vnd.apple.mpegurl')) {
    report({ state: 'loading' })
    video.src = url

    const handleLoadedMetadata = () => {
      report({ state: 'ready' })
    }
    const handleError = () => {
      report({ state: 'error', message: 'Playback error' })
    }

    video.addEventListener('loadedmetadata', handleLoadedMetadata)
    video.addEventListener('error', handleError)

    return () => {
      destroyed = true
      video.removeEventListener('loadedmetadata', handleLoadedMetadata)
      video.removeEventListener('error', handleError)
      video.removeAttribute('src')
      video.load()
    }
  }

  report({ state: 'error', message: 'HLS is not supported in this browser.' })
  return () => {
    destroyed = true
  }
}
