import { useEffect, useMemo, useState } from 'react'
import { useLocation } from 'react-router-dom'

import HlsPlayer from '../components/HlsPlayer'
import { resolveBackendUrl } from '../lib/backendUrl'
import type { PlayerConfig } from '../types/playerConfig'

const EMPTY_CONFIG: PlayerConfig = {
  url: '',
  autoplay: false,
  muted: false,
  loop: false,
  poster: '',
  controls: true,
  preload: 'metadata',
  proxy: false,
}

function EmbedPage() {
  const location = useLocation()
  const [config, setConfig] = useState<PlayerConfig>(EMPTY_CONFIG)
  const [status, setStatus] = useState<'idle' | 'loading' | 'ready' | 'error'>('idle')
  const [error, setError] = useState<string | null>(null)
  const token = useMemo(() => {
    const params = new URLSearchParams(location.search)
    return params.get('token')
  }, [location.search])

  useEffect(() => {
    let active = true

    const loadTokenConfig = async () => {
      if (!token) {
        setStatus('error')
        setError('Missing embed token.')
        return
      }
      setStatus('loading')
      setError(null)
      try {
        const backendBase = resolveBackendUrl().replace(/\/$/, '')
        const response = await fetch(`${backendBase}/embed/token/${encodeURIComponent(token)}`)
        if (!response.ok) {
          if (response.status === 410) {
            throw new Error('Embed token expired.')
          }
          if (response.status === 404) {
            throw new Error('Embed token not found.')
          }
          throw new Error('Unable to load embed token.')
        }
        const data = await response.json()
        if (!active) return
        if (!data?.config) {
          throw new Error('Invalid embed token response.')
        }
        setConfig(data.config as PlayerConfig)
        setStatus('ready')
      } catch (err) {
        if (!active) return
        setStatus('error')
        setError(err instanceof Error ? err.message : 'Unable to load embed token.')
      }
    }

    loadTokenConfig()

    return () => {
      active = false
    }
  }, [token])

  return (
    <div className="embed-page">
      <div className="embed-stage">
        <div className="preview-frame">
          {status === 'loading' && (
            <div className="embed-status">Loading embed…</div>
          )}
          {status === 'error' && (
            <div className="embed-error">{error}</div>
          )}
          {status === 'ready' && config.url && (
            <HlsPlayer config={config} />
          )}
        </div>
      </div>
    </div>
  )
}

export default EmbedPage
