import React from 'react'
import type { ReactElement } from 'react'
import { render, screen, waitFor } from '@testing-library/react'
import userEvent from '@testing-library/user-event'
import { MemoryRouter, useLocation } from 'react-router-dom'
import { beforeEach, describe, expect, it, vi } from 'vitest'

import ToolboxPage from './ToolboxPage'

vi.mock('../components/HlsPlayer', () => ({
  default: () => <div data-testid="hls-player" />,
}))

vi.mock('../components/EmbedSnippet', () => ({
  default: ({ open }: { open: boolean }) =>
    open ? <div data-testid="embed-snippet" /> : null,
}))

describe('ToolboxPage', () => {
  const fetchMock = vi.fn()
  const LocationProbe = () => {
    const location = useLocation()
    return <div data-testid="location-search">{location.search}</div>
  }
  const renderWithRouter = (ui: ReactElement) => render(
    <MemoryRouter>
      {ui}
      <LocationProbe />
    </MemoryRouter>
  )

  beforeEach(() => {
    fetchMock.mockReset()
    const storage = window.localStorage
    if (storage && typeof storage.clear === 'function') {
      storage.clear()
    }
    fetchMock.mockImplementation((input: RequestInfo) => {
      const url = String(input)
      if (url.includes('/admin/me')) {
        return Promise.resolve({
          ok: true,
          json: async () => ({ authenticated: false }),
        })
      }
      if (url.includes('/local/config')) {
        return Promise.resolve({
          ok: false,
          json: async () => ({}),
        })
      }
      return Promise.resolve({
        ok: true,
        json: async () => ({}),
      })
    })
    vi.stubGlobal('fetch', fetchMock)
  })

  it('renders admin login controls when local VOD is enabled', async () => {
    // Purpose: ensure the admin-only local VOD flow prompts for credentials when enabled.
    fetchMock.mockImplementation((input: RequestInfo) => {
      const url = String(input)
      if (url.includes('/local/config')) {
        return Promise.resolve({
          ok: true,
          json: async () => ({ enabled: true, defaultTtlSeconds: 3600, maxTtlSeconds: 7200 }),
        })
      }
      if (url.includes('/admin/me')) {
        return Promise.resolve({
          ok: true,
          json: async () => ({ authenticated: false }),
        })
      }
      return Promise.resolve({ ok: true, json: async () => ({}) })
    })

    renderWithRouter(<ToolboxPage />)
    expect(await screen.findByPlaceholderText('ADMIN_USER')).toBeInTheDocument()
    expect(screen.getByPlaceholderText('ADMIN_PASS')).toBeInTheDocument()
  })

  it('pushes config updates to the URL query string', async () => {
    // Purpose: ensure the main source input still syncs to the query string.
    const user = userEvent.setup()
    renderWithRouter(<ToolboxPage />)

    const input = screen.getByLabelText(/source_url/i)
    await user.type(input, 'https://example.com/stream.m3u8')
    await waitFor(() => {
      expect(screen.getByTestId('location-search').textContent).toContain('stream.m3u8')
    })
  })

  it('opens the embed snippet when publish is clicked', async () => {
    // Purpose: ensure embed export is still available after local VOD UI changes.
    const user = userEvent.setup()
    renderWithRouter(<ToolboxPage />)

    const input = screen.getByLabelText(/source_url/i)
    await user.type(input, 'https://example.com/stream.m3u8')
    await new Promise((resolve) => setTimeout(resolve, 700))

    const publish = screen.getByRole('button', { name: /export_embed/i })
    await user.click(publish)

    expect(screen.getByText(/embed_code/i)).toBeInTheDocument()
  })
})
