import { useCallback, useEffect, useMemo, useRef, useState } from 'react'
import { NavLink, useLocation, useNavigate } from 'react-router-dom'
import { resolveBackendUrl } from '../lib/backendUrl'
import { decodeConfig, encodeConfig } from '../lib/configCodec'
import { isGoogleVideoUrl } from '../lib/urlRules'
import { DEFAULT_PLAYER_CONFIG } from '../types/playerConfig'
import type { HlsStatus } from '../lib/hlsController'
import type { PlayerConfig } from '../types/playerConfig'

import HlsPlayer from '../components/HlsPlayer'
import { FileBrowserDialog } from '../components/toolbox/FileBrowserDialog'
import { UploadedPostersDialog } from '../components/toolbox/UploadedPostersDialog'

// --- Types ---

type LocalTokenKind = 'localhls' | 'localfile'

type LocalShare = {
  shareId: string
  path: string
  kind: 'hls' | 'file'
  createdAt: number
  expiresAt: number
  posterUrl?: string | null
}

type LocalBrowseEntry = {
  name: string
  type: 'dir' | 'file'
  ext?: string
}

type LocalConfig = {
  enabled: boolean
  defaultTtlSeconds: number
  maxTtlSeconds: number
}

type EmbedToken = {
  tokenId: string
  config: PlayerConfig
  createdAt: number
  expiresAt: number | null
  ttlSeconds: number | null
  usageCount: number
  lastAccessedAt: number | null
  description?: string | null
}

type UploadEntry = {
  name: string
  url: string
  updatedAt: number
}

type ToolboxPersistedState = {
  version: 1
  config: PlayerConfig
  sourceInput: string
  showEmbedCode: boolean
  embedSize: 'responsive' | 'fixed-medium' | 'fixed-small'
  shareTtlHours: number
  autoProxyOptOutUrl: string | null
  adminStatus: 'authenticated' | 'guest'
}

// --- Helpers ---

const isRemoteUrl = (value: string) => /^https?:\/\//i.test(value)

const isSameConfig = (left: PlayerConfig, right: PlayerConfig) => (
  left.url === right.url &&
  left.autoplay === right.autoplay &&
  left.muted === right.muted &&
  left.loop === right.loop &&
  left.poster === right.poster &&
  left.controls === right.controls &&
  left.preload === right.preload &&
  left.proxy === right.proxy
)

const TOOLBOX_STORAGE_KEY = 'hls-player.toolbox-state'
const STORAGE_VERSION = 1

const readPersistedState = (): ToolboxPersistedState | null => {
  if (typeof window === 'undefined') return null
  try {
    const raw = window.localStorage.getItem(TOOLBOX_STORAGE_KEY)
    if (!raw) return null
    const parsed = JSON.parse(raw) as ToolboxPersistedState
    if (!parsed || parsed.version !== STORAGE_VERSION) return null
    return parsed
  } catch {
    return null
  }
}

const writePersistedState = (state: ToolboxPersistedState) => {
  if (typeof window === 'undefined') return
  try {
    window.localStorage.setItem(TOOLBOX_STORAGE_KEY, JSON.stringify(state))
  } catch {
    // ignore storage failures
  }
}

const parseLocalToken = (value: string) => {
  if (!value) return null
  const [prefix, shareId] = value.split(':')
  if (!shareId) return null
  if (prefix !== 'localhls' && prefix !== 'localfile') return null
  return { kind: prefix as LocalTokenKind, shareId }
}

const ALLOWED_IMAGE_EXTENSIONS = new Set([
  '.jpeg',
  '.jpg',
  '.png',
  '.gif',
  '.webp',
  '.tiff',
  '.tif',
  '.bmp',
])

const isAllowedImage = (name: string) => {
  const extension = name.slice(name.lastIndexOf('.')).toLowerCase()
  if (!extension || extension === name) return false
  return ALLOWED_IMAGE_EXTENSIONS.has(extension)
}

// --- Icons ---
const IconPlay = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><polygon points="5 3 19 12 5 21 5 3"/></svg>
const IconMute = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M11 5 6 9H2v6h4l5 4V5Z"/><line x1="23" x2="17" y1="9" y2="15"/><line x1="17" x2="23" y1="9" y2="15"/></svg>
const IconRepeat = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="m17 2 4 4-4 4"/><path d="M3 11v-1a4 4 0 0 1 4-4h14"/><path d="m7 22-4-4 4-4"/><path d="M21 13v1a4 4 0 0 1-4 4H3"/></svg>
const IconGlobe = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><circle cx="12" cy="12" r="10"/><path d="M12 2a14.5 14.5 0 0 0 0 20 14.5 14.5 0 0 0 0-20"/><path d="M2 12h20"/></svg>
const IconSettings = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1Z"/></svg>
const IconFolder = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M20 20a2 2 0 0 0 2-2V8a2 2 0 0 0-2-2h-7.9a2 2 0 0 1-1.69-.9L9.6 3.9A2 2 0 0 0 7.93 3H4a2 2 0 0 0-2 2v13a2 2 0 0 0 2 2Z"/></svg>
const IconCode = () => <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><polyline points="16 18 22 12 16 6"/><polyline points="8 6 2 12 8 18"/></svg>

// --- Components ---

function SectionHeader({ title }: { title: string }) {
  return (
    <h3 className="text-xs font-bold text-zinc-500 tracking-wider mb-4 flex items-center gap-2 uppercase">
      {title}
    </h3>
  )
}

function ToggleOption({ 
  label, 
  active, 
  onClick, 
  icon: Icon,
  warning = false
}: { 
  label: string, 
  active: boolean, 
  onClick: () => void,
  icon?: React.FC,
  warning?: boolean
}) {
  return (
    <button
      onClick={onClick}
      className={`
        flex items-center justify-between p-3 rounded-lg border transition-all duration-200
        ${active 
          ? (warning ? 'bg-red-500/10 border-red-500/50 text-red-500' : 'bg-accent/10 border-accent/50 text-accent') 
          : 'bg-void-border/50 border-void-border text-zinc-400 hover:border-zinc-700 hover:bg-void-border'}
      `}
    >
      <div className="flex items-center gap-3">
        {Icon && <Icon />}
        <span className="text-xs font-bold tracking-wide">{label}</span>
      </div>
      <div className={`w-2 h-2 rounded-full ${active ? (warning ? 'bg-red-500 shadow-[0_0_8px_rgba(218,37,29,0.5)]' : 'bg-accent shadow-[0_0_8px_rgba(255,217,0,0.5)]') : 'bg-zinc-700'}`} />
    </button>
  )
}

function QuickToggle({
  label,
  active,
  onClick,
  icon: Icon,
  warning = false,
}: {
  label: string
  active: boolean
  onClick: () => void
  icon?: React.FC
  warning?: boolean
}) {
  return (
    <button
      onClick={onClick}
      className={`
        flex items-center gap-2 px-3 py-1.5 rounded-full border text-[10px] font-bold tracking-[0.2em] transition-all
        ${active
          ? (warning ? 'bg-red-500/10 border-red-500/60 text-red-400' : 'bg-accent/10 border-accent/60 text-accent')
          : 'bg-void-border/60 border-void-border text-zinc-400 hover:border-zinc-600 hover:text-zinc-200'}
      `}
    >
      {Icon && <Icon />}
      {label}
      <span className={`h-1.5 w-1.5 rounded-full ${active ? (warning ? 'bg-red-400' : 'bg-accent') : 'bg-zinc-700'}`} />
    </button>
  )
}

export default function ToolboxPage() {
  const location = useLocation()
  const navigate = useNavigate()
  const persistedState = useMemo(() => readPersistedState(), [])
  const resolveConfigFromSearch = useCallback((search: string) => {
    const params = new URLSearchParams(search)
    const decoded = decodeConfig(params)
    return decoded.url ? decoded : persistedState?.config || DEFAULT_PLAYER_CONFIG
  }, [persistedState])
  // --- State Initialization ---
  const initialConfig = useMemo(() => {
    return resolveConfigFromSearch(location.search)
  }, [location.search, resolveConfigFromSearch])

  const backendBase = useMemo(() => resolveBackendUrl().replace(/\/$/, ''), [])

  const [config, setConfig] = useState(initialConfig)
  const [sourceInput, setSourceInput] = useState(() => (
    persistedState?.sourceInput || initialConfig.url
  ))
  const [status, setStatus] = useState<HlsStatus>({ state: 'idle' })
  const [showEmbedCode, setShowEmbedCode] = useState(persistedState?.showEmbedCode || false)
  const [embedSize, setEmbedSize] = useState<'responsive' | 'fixed-medium' | 'fixed-small'>(
    persistedState?.embedSize || 'responsive'
  )
  const [copyFeedback, setCopyFeedback] = useState(false)
  const [isUploading, setIsUploading] = useState(false)
  const [uploadedPosters, setUploadedPosters] = useState<UploadEntry[]>([])
  const [uploadsOpen, setUploadsOpen] = useState(false)
  const [uploadsLoading, setUploadsLoading] = useState(false)
  const [uploadsError, setUploadsError] = useState<string | null>(null)
  
  // Local/Admin State
  const [localConfig, setLocalConfig] = useState<LocalConfig>({
    enabled: false,
    defaultTtlSeconds: 24 * 60 * 60,
    maxTtlSeconds: 30 * 24 * 60 * 60,
  })
  const [adminStatus, setAdminStatus] = useState<'checking' | 'authenticated' | 'guest'>(
    persistedState?.adminStatus || 'checking'
  )
  const [adminError, setAdminError] = useState<string | null>(null)
  const [loginForm, setLoginForm] = useState({ username: '', password: '' })
  const [shareTtlHours, setShareTtlHours] = useState(persistedState?.shareTtlHours || 24)
  const [shares, setShares] = useState<LocalShare[]>([])
  const [isShareBusy, setIsShareBusy] = useState(false)
  const [sharePosterBusyId, setSharePosterBusyId] = useState<string | null>(null)
  const [sharePosterTarget, setSharePosterTarget] = useState<LocalShare | null>(null)
  const [browseState, setBrowseState] = useState<{
    open: boolean
    dir: string
    entries: LocalBrowseEntry[]
    loading: boolean
    error: string | null
  }>({
    open: false,
    dir: '',
    entries: [],
    loading: false,
    error: null,
  })
  const [localError, setLocalError] = useState<string | null>(null)
  const [embedTokens, setEmbedTokens] = useState<EmbedToken[]>([])
  const [embedTokensLoading, setEmbedTokensLoading] = useState(false)
  const [embedTokensError, setEmbedTokensError] = useState<string | null>(null)
  const [embedCreateBusy, setEmbedCreateBusy] = useState(false)
  const [embedTokenDescription, setEmbedTokenDescription] = useState('')
  const [embedTtlMode, setEmbedTtlMode] = useState<'preset' | 'custom'>('preset')
  const [embedTtlPreset, setEmbedTtlPreset] = useState('3h')
  const [embedCustomTtlValue, setEmbedCustomTtlValue] = useState(3)
  const [embedCustomTtlUnit, setEmbedCustomTtlUnit] = useState<'hours' | 'days'>('hours')
  const [selectedEmbedTokenId, setSelectedEmbedTokenId] = useState<string | null>(null)
  const fileInputRef = useRef<HTMLInputElement>(null)
  const embedSizeOptions = ['responsive', 'fixed-medium', 'fixed-small'] as const
  const embedTtlPresets = {
    hours: ['3h', '6h', '12h', '24h'],
    days: ['1d', '3d', '5d', '7d', '10d', '14d', '30d'],
  }
  
  // Auto-Proxy Logic
  const [autoProxyOptOutUrl, setAutoProxyOptOutUrl] = useState<string | null>(
    persistedState?.autoProxyOptOutUrl || null
  )
  const prevProxyRef = useRef(config.proxy)
  const localToken = useMemo(() => parseLocalToken(config.url), [config.url])
  const activeShareId = localToken?.shareId || null

  // --- Effects ---

  useEffect(() => {
    const params = encodeConfig(config)
    const query = params.toString()
    const nextSearch = `?${query}`
    if (location.search === nextSearch) {
      return
    }
    navigate({ pathname: location.pathname, search: nextSearch })
  }, [config, location.pathname, location.search, navigate])

  useEffect(() => {
    const nextConfig = resolveConfigFromSearch(location.search)
    setConfig((prev) => (isSameConfig(prev, nextConfig) ? prev : nextConfig))
  }, [location.search, resolveConfigFromSearch])

  useEffect(() => {
    if (!localToken) {
      setSourceInput(config.url)
    }
  }, [config.url, localToken])

  useEffect(() => {
    if (!localToken) return
    const match = shares.find((share) => share.shareId === localToken.shareId)
    if (match) {
      setSourceInput(match.path)
      const nextPoster = match.posterUrl || ''
      setConfig((prev) => (prev.poster === nextPoster ? prev : { ...prev, poster: nextPoster }))
    }
  }, [localToken, shares])

  useEffect(() => {
    let active = true

    const loadLocalConfig = async () => {
      try {
        const response = await fetch(`${backendBase}/local/config`)
        if (!response.ok) throw new Error('local_disabled')
        const data = await response.json()
        if (!active) return
        setLocalConfig({
          enabled: Boolean(data.enabled),
          defaultTtlSeconds: Number(data.defaultTtlSeconds || 24 * 60 * 60),
          maxTtlSeconds: Number(data.maxTtlSeconds || 30 * 24 * 60 * 60),
        })
        setShareTtlHours(Math.max(1, Math.round(Number(data.defaultTtlSeconds || 24 * 60 * 60) / 3600)))
      } catch {
        if (!active) return
        setLocalConfig((prev) => ({ ...prev, enabled: false }))
      }
    }

    const checkAdmin = async () => {
      try {
        const response = await fetch(`${backendBase}/admin/me`, { credentials: 'include' })
        if (!response.ok) throw new Error('not_authenticated')
        const data = await response.json()
        if (!active) return
        setAdminStatus(data.authenticated ? 'authenticated' : 'guest')
      } catch {
        if (!active) return
        setAdminStatus('guest')
      }
    }

    loadLocalConfig()
    checkAdmin()

    return () => { active = false }
  }, [backendBase])

  useEffect(() => {
    const isGoogle = isGoogleVideoUrl(config.url)
    if (isGoogle && !config.proxy && autoProxyOptOutUrl !== config.url) {
      setConfig((prev) => {
        if (!isGoogleVideoUrl(prev.url) || prev.proxy) return prev
        if (autoProxyOptOutUrl === prev.url) return prev
        return { ...prev, proxy: true }
      })
    }
    if (!isGoogle && autoProxyOptOutUrl) {
      setAutoProxyOptOutUrl(null)
    }
  }, [config.url, config.proxy, autoProxyOptOutUrl])

  useEffect(() => {
    const isGoogle = isGoogleVideoUrl(config.url)
    if (prevProxyRef.current && !config.proxy && isGoogle) {
      setAutoProxyOptOutUrl(config.url || null)
    }
    if (config.proxy && autoProxyOptOutUrl === config.url) {
      setAutoProxyOptOutUrl(null)
    }
    prevProxyRef.current = config.proxy
  }, [config.proxy, config.url, autoProxyOptOutUrl])

  useEffect(() => {
    if (!config.url) setStatus({ state: 'idle' })
  }, [config.url])

  useEffect(() => {
    const nextState: ToolboxPersistedState = {
      version: STORAGE_VERSION,
      config,
      sourceInput,
      showEmbedCode,
      embedSize,
      shareTtlHours,
      autoProxyOptOutUrl,
      adminStatus: adminStatus === 'authenticated' ? 'authenticated' : 'guest',
    }
    writePersistedState(nextState)
  }, [
    config,
    sourceInput,
    showEmbedCode,
    embedSize,
    shareTtlHours,
    autoProxyOptOutUrl,
    adminStatus,
  ])

  // --- Methods ---

  const embedUrl = useMemo(() => {
    if (!selectedEmbedTokenId) {
      return ''
    }
    return `${window.location.origin}/embed?token=${selectedEmbedTokenId}`
  }, [selectedEmbedTokenId])

  const embedSnippet = useMemo(() => {
    if (!embedUrl) {
      return 'Create a token to generate an embed snippet.'
    }
    if (embedSize === 'fixed-medium') {
      return `<iframe src="${embedUrl}" width="640" height="360" allow="autoplay; fullscreen; picture-in-picture" loading="lazy" frameborder="0"></iframe>`
    }
    if (embedSize === 'fixed-small') {
      return `<iframe src="${embedUrl}" width="480" height="270" allow="autoplay; fullscreen; picture-in-picture" loading="lazy" frameborder="0"></iframe>`
    }
    return [
      '<div style="position:relative;width:100%;padding-top:56.25%;">',
      `  <iframe src="${embedUrl}" style="position:absolute;top:0;left:0;width:100%;height:100%;" allow="autoplay; fullscreen; picture-in-picture" loading="lazy" frameborder="0"></iframe>`,
      '</div>',
    ].join('\n')
  }, [embedUrl, embedSize])

  const handleCopy = () => {
    navigator.clipboard.writeText(embedSnippet)
    setCopyFeedback(true)
    setTimeout(() => setCopyFeedback(false), 2000)
  }

  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (!file) return

    setIsUploading(true)
    const formData = new FormData()
    formData.append('file', file)

    try {
      const backendUrl = resolveBackendUrl()
      const response = await fetch(`${backendUrl.replace(/\/$/, '')}/upload`, {
        method: 'POST',
        body: formData,
      })

      if (!response.ok) throw new Error('Upload failed')

      const data = await response.json()
      setConfig((prev) => ({ ...prev, poster: data.url }))
      const name = typeof data.url === 'string' ? data.url.split('/').pop() || data.url : file.name
      setUploadedPosters((prev) => {
        const next = [{ name, url: data.url, updatedAt: Math.floor(Date.now() / 1000) }, ...prev]
        const unique = new Map(next.map((entry) => [entry.url, entry]))
        return Array.from(unique.values())
      })
    } catch (error) {
      console.error('Error uploading poster:', error)
      alert('Failed to upload poster image')
    } finally {
      setIsUploading(false)
      if (fileInputRef.current) fileInputRef.current.value = ''
    }
  }

  const loadUploads = useCallback(async () => {
    setUploadsLoading(true)
    setUploadsError(null)
    try {
      let uploadsUrl = ''
      try {
        uploadsUrl = new URL('/uploads/list', backendBase).toString()
      } catch {
        setUploadsError('Invalid backend URL')
        setUploadsLoading(false)
        return
      }
      const response = await fetch(uploadsUrl)
      if (!response.ok) throw new Error('uploads_failed')
      const contentType = response.headers.get('content-type') || ''
      if (!contentType.includes('application/json')) {
        throw new Error('uploads_invalid')
      }
      const data = await response.json()
      const files = Array.isArray(data.files) ? (data.files as UploadEntry[]) : []
      setUploadedPosters(files.filter((entry: UploadEntry) => isAllowedImage(entry.name)))
    } catch (error) {
      console.error('Failed to load uploads:', error)
      setUploadsError('Failed to load uploads')
    } finally {
      setUploadsLoading(false)
    }
  }, [backendBase])

  const handleOpenUploads = async () => {
    setUploadsOpen(true)
    await loadUploads()
  }

  const handleCloseUploads = () => {
    setSharePosterTarget(null)
    setUploadsOpen(false)
  }

  const handleSelectUpload = (entry: UploadEntry) => {
    if (sharePosterTarget) {
      const target = sharePosterTarget
      setSharePosterTarget(null)
      setUploadsOpen(false)
      setShares((prev) => prev.map((item) => (
        item.shareId === target.shareId
          ? { ...item, posterUrl: entry.url }
          : item
      )))
      if (activeShareId === target.shareId) {
        setConfig((prev) => ({ ...prev, poster: entry.url }))
      }
      handleAssignSharePoster(target, entry.url)
      return
    }
    setConfig((prev) => ({ ...prev, poster: entry.url }))
    setUploadsOpen(false)
  }

  const adminFetch = useCallback(
    (path: string, options: RequestInit = {}) => fetch(`${backendBase}${path}`, {
      ...options,
      credentials: 'include',
    }),
    [backendBase],
  )

  const loadShares = useCallback(async () => {
    try {
      const response = await adminFetch('/local/shares')
      if (!response.ok) throw new Error('share_load_failed')
      const data = await response.json()
      setShares(data.shares || [])
      setLocalError(null)
    } catch (error) {
      console.error('Error loading shares:', error)
      setLocalError('Failed to load shares')
    }
  }, [adminFetch])

  const loadEmbedTokens = useCallback(async () => {
    setEmbedTokensLoading(true)
    setEmbedTokensError(null)
    try {
      const response = await adminFetch('/embed/tokens')
      if (!response.ok) throw new Error('embed_tokens_failed')
      const data = await response.json()
      const tokens = Array.isArray(data.tokens) ? (data.tokens as EmbedToken[]) : []
      setEmbedTokens(tokens)
    } catch (error) {
      console.error('Error loading embed tokens:', error)
      setEmbedTokensError('Failed to load embed tokens')
    } finally {
      setEmbedTokensLoading(false)
    }
  }, [adminFetch])

  useEffect(() => {
    if (adminStatus === 'authenticated') {
      loadShares()
      loadEmbedTokens()
    }
  }, [adminStatus, loadEmbedTokens, loadShares])

  const handleLogin = async () => {
    setAdminError(null)
    try {
      const response = await adminFetch('/admin/login', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          username: loginForm.username,
          password: loginForm.password,
        }),
      })
      if (!response.ok) throw new Error('login_failed')
      setAdminStatus('authenticated')
      setAdminError(null)
      setLoginForm({ username: '', password: '' })
      loadShares()
    } catch (error) {
      console.error('Admin login failed:', error)
      setAdminError('Login failed')
    }
  }

  const handleLogout = async () => {
    try {
      await adminFetch('/admin/logout', { method: 'POST' })
    } catch (error) {
      console.error('Admin logout failed:', error)
    } finally {
      setAdminStatus('guest')
      setShares([])
    }
  }

  const openBrowse = async (dir = '') => {
    setBrowseState((prev) => ({ ...prev, open: true, loading: true, error: null, dir }))
    try {
      const response = await adminFetch(`/local/browse?dir=${encodeURIComponent(dir)}`)
      if (!response.ok) throw new Error('browse_failed')
      const data = await response.json()
      setBrowseState({ open: true, loading: false, error: null, dir: data.dir || '', entries: data.entries || [] })
    } catch (error) {
      console.error('Browse failed:', error)
      setBrowseState((prev) => ({ ...prev, loading: false, error: 'Failed to load directory' }))
    }
  }

  const selectBrowseEntry = (entry: LocalBrowseEntry) => {
    if (entry.type === 'dir') {
      openBrowse(browseState.dir ? `${browseState.dir}/${entry.name}` : entry.name)
      return
    }
    const nextPath = browseState.dir ? `${browseState.dir}/${entry.name}` : entry.name
    setSourceInput(nextPath)
    setBrowseState(prev => ({ ...prev, open: false }))
  }

  const handleCreateShare = async () => {
    setLocalError(null)
    const pathValue = sourceInput.trim()
    if (!pathValue || isRemoteUrl(pathValue)) {
      setLocalError('Enter a relative path to a local file')
      return
    }
    const requestedSeconds = Math.max(1, Math.round(shareTtlHours)) * 60 * 60
    const ttlSeconds = Math.min(requestedSeconds, localConfig.maxTtlSeconds || requestedSeconds)
    setIsShareBusy(true)
    try {
      const response = await adminFetch('/local/shares', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          path: pathValue,
          ttlSeconds,
          posterUrl: config.poster || null,
        }),
      })
      if (!response.ok) throw new Error('share_failed')
      const data = await response.json()
      setConfig((prev) => ({
        ...prev,
        url: data.token,
        poster: typeof data.posterUrl === 'string' ? data.posterUrl : (prev.poster || ''),
      }))
      setSourceInput(data.path)
      setShares((prev) => {
        const created: LocalShare = {
          shareId: data.shareId,
          path: data.path,
          kind: data.kind,
          createdAt: Math.floor(Date.now() / 1000),
          expiresAt: data.expiresAt,
          posterUrl: data.posterUrl || null,
        }
        const next = [created, ...prev]
        const unique = new Map(next.map((entry) => [entry.shareId, entry]))
        return Array.from(unique.values())
      })
      loadShares()
    } catch (error) {
      console.error('Create share failed:', error)
      setLocalError('Failed to create share')
    } finally {
      setIsShareBusy(false)
    }
  }

  const resolveEmbedTokenTtlPayload = () => {
    if (embedTtlMode === 'custom') {
      return {
        customTtlValue: embedCustomTtlValue,
        customTtlUnit: embedCustomTtlUnit,
      }
    }
    return { ttlPreset: embedTtlPreset }
  }

  const handleCreateEmbedToken = async () => {
    setEmbedTokensError(null)
    if (embedCreateBusy) {
      return
    }
    setEmbedCreateBusy(true)
    try {
      const response = await adminFetch('/embed/tokens', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          config,
          description: embedTokenDescription,
          ...resolveEmbedTokenTtlPayload(),
        }),
      })
      if (!response.ok) throw new Error('embed_create_failed')
      const data = await response.json()
      setSelectedEmbedTokenId(data.tokenId)
      setEmbedTokenDescription('')
      loadEmbedTokens()
    } catch (error) {
      console.error('Create embed token failed:', error)
      setEmbedTokensError('Failed to create embed token')
    } finally {
      setEmbedCreateBusy(false)
    }
  }

  const handleRevokeEmbedToken = async (tokenId: string) => {
    setEmbedTokensError(null)
    try {
      await adminFetch(`/embed/tokens/${tokenId}`, { method: 'DELETE' })
      if (selectedEmbedTokenId === tokenId) {
        setSelectedEmbedTokenId(null)
      }
      loadEmbedTokens()
    } catch (error) {
      console.error('Revoke embed token failed:', error)
      setEmbedTokensError('Failed to revoke embed token')
    }
  }

  const handleRevokeShare = async (shareId: string) => {
    try {
      await adminFetch(`/local/shares/${shareId}`, { method: 'DELETE' })
      if (localToken?.shareId === shareId) {
        setConfig((prev) => ({ ...prev, url: '' }))
      }
      loadShares()
    } catch (error) {
      console.error('Revoke share failed:', error)
      setLocalError('Failed to revoke share')
    }
  }

  const handleUseShare = (share: LocalShare) => {
    const token = share.kind === 'hls' ? `localhls:${share.shareId}` : `localfile:${share.shareId}`
    setConfig((prev) => ({ ...prev, url: token, poster: share.posterUrl || '' }))
    setSourceInput(share.path)
  }

  const handleAssignSharePoster = async (share: LocalShare, posterUrl: string) => {
    setSharePosterBusyId(share.shareId)
    try {
      const response = await adminFetch(`/local/shares/${share.shareId}`, {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ posterUrl }),
      })
      if (!response.ok) throw new Error('share_update_failed')
      const data = await response.json()
      const nextPoster = data.share?.posterUrl || posterUrl
      setShares((prev) => prev.map((item) => (
        item.shareId === share.shareId
          ? { ...item, posterUrl: data.share?.posterUrl || posterUrl }
          : item
      )))
      if (localToken?.shareId === share.shareId) {
        setConfig((prev) => ({ ...prev, poster: nextPoster }))
      }
    } catch (error) {
      console.error('Update share poster failed:', error)
      setLocalError('Failed to update share poster')
      loadShares()
    } finally {
      setSharePosterBusyId((prev) => (prev === share.shareId ? null : prev))
    }
  }

  return (
    <div className="flex h-screen w-full bg-void-black text-white overflow-hidden font-sans flex-col">
      <div className="flex flex-1 min-h-0">
        {/* --- Left Panel: Player (Responsive) --- */}
        <div className="flex-1 flex flex-col min-w-0 bg-void-black relative">
         
         {/* Navigation Bar */}
         <div className="absolute top-0 left-0 right-0 z-20 p-6 flex justify-between items-start pointer-events-none">
            <div className="pointer-events-auto flex items-center gap-6">
              <div className="flex items-center gap-2">
                <span className="w-2 h-2 rounded-full bg-vn-yellow shadow-[0_0_8px_rgba(255,217,0,0.6)] animate-pulse"></span>
                <h1 className="text-xl font-bold tracking-tight text-white font-mono">HLS<span className="text-vn-yellow">.IO</span></h1>
              </div>
              
               <nav className="flex bg-void-panel/90 backdrop-blur rounded-full p-1 border border-void-border shadow-xl">
                 <NavLink 
                   to="/" 
                   end 
                   className={({ isActive }) => `px-4 py-1.5 rounded-full text-xs font-bold transition-all ${isActive ? 'bg-vn-red text-vn-yellow shadow-lg shadow-vn-red/30' : 'text-zinc-400 hover:text-white'}`}
                 >
                   PLAYER
                 </NavLink>
               </nav>
            </div>
            
            <div className="pointer-events-auto">
               <div className={`
                 px-3 py-1.5 rounded-md text-[10px] font-mono font-bold uppercase tracking-wider border
                 ${status.state === 'ready' ? 'bg-accent/10 border-accent/20 text-accent' : 
                   status.state === 'error' ? 'bg-red-500/10 border-red-500/20 text-red-400' : 'bg-void-border border-void-border text-zinc-500'}
               `}>
                 {status.state}
               </div>
            </div>
         </div>

         {/* Player Display Area */}
         <div className="flex-1 flex items-center justify-center p-4 lg:p-12 relative">
            <div className="absolute inset-0 bg-[radial-gradient(circle_at_center,_var(--tw-gradient-stops))] from-void-panel via-void-black to-void-black opacity-50 pointer-events-none" />
            
            <div className="w-full max-w-6xl flex flex-col items-center gap-4 z-10">
              <div className="pointer-events-auto flex flex-wrap items-center justify-center gap-2 rounded-full border border-void-border bg-void-panel/80 px-3 py-2 shadow-xl">
                <QuickToggle
                  label="AUTO"
                  active={config.autoplay}
                  onClick={() => setConfig(prev => ({ ...prev, autoplay: !prev.autoplay }))}
                  icon={IconPlay}
                />
                <QuickToggle
                  label="MUTE"
                  active={config.muted}
                  onClick={() => setConfig(prev => ({ ...prev, muted: !prev.muted }))}
                  icon={IconMute}
                />
                <QuickToggle
                  label="LOOP"
                  active={config.loop}
                  onClick={() => setConfig(prev => ({ ...prev, loop: !prev.loop }))}
                  icon={IconRepeat}
                />
                <QuickToggle
                  label="PROXY"
                  active={config.proxy}
                  onClick={() => setConfig(prev => ({ ...prev, proxy: !prev.proxy }))}
                  icon={IconGlobe}
                  warning
                />
                <QuickToggle
                  label="UI"
                  active={config.controls}
                  onClick={() => setConfig(prev => ({ ...prev, controls: !prev.controls }))}
                  icon={IconSettings}
                />
              </div>
              <div className="w-full aspect-video shadow-2xl rounded-2xl overflow-hidden ring-1 ring-void-border relative bg-black transition-all duration-500 ease-out">
                {config.url ? (
                  <HlsPlayer config={config} onStatusChange={setStatus} />
                ) : (
                  <div className="absolute inset-0 flex flex-col items-center justify-center text-zinc-700 select-none">
                    <div className="text-6xl mb-4 opacity-20">
                      <IconPlay />
                    </div>
                    <p className="font-mono text-sm tracking-[0.2em] opacity-50">NO SIGNAL SOURCE</p>
                  </div>
                )}
              </div>
            </div>
         </div>
        </div>

        {/* --- Right Panel: Controls --- */}
        <div className="w-full lg:w-[380px] xl:w-[460px] bg-void-panel border-l border-void-border flex flex-col h-full shadow-2xl z-30 relative">
          <div className="flex-1 overflow-y-auto overflow-x-hidden p-6 space-y-8 custom-scrollbar">
           
           {/* Section: Source */}
           <section>
              <SectionHeader title="SOURCE FEED" />
              <div className="bg-void-black p-1.5 rounded-xl border border-void-border focus-within:border-accent/40 focus-within:bg-void-black transition-all shadow-inner">
                <div className="relative">
                  <label htmlFor="source_url" className="sr-only">SOURCE_URL</label>
                  <input
                    id="source_url"
                    className="w-full bg-transparent border-none text-sm text-white px-3 py-2.5 placeholder-zinc-600 focus:ring-0 font-mono"
                    placeholder="https://..."
                    value={sourceInput}
                    onChange={(e) => {
                       const nextValue = e.target.value
                       setSourceInput(nextValue)
                       if (!nextValue) {
                         setConfig((prev) => ({ ...prev, url: '' }))
                         return
                       }
                       if (isRemoteUrl(nextValue)) {
                         setConfig((prev) => ({ ...prev, url: nextValue }))
                         return
                       }
                       setConfig((prev) => (prev.url ? { ...prev, url: '' } : prev))
                    }}
                    spellCheck={false}
                  />
                  <div className="absolute right-3 top-1/2 -translate-y-1/2 text-[10px] font-bold px-1.5 py-0.5 rounded bg-void-border text-zinc-400 border border-void-border pointer-events-none">
                    {sourceInput && !isRemoteUrl(sourceInput) ? 'LOCAL' : 'URL'}
                  </div>
                </div>
              </div>
           </section>

           {/* Section: Controls */}
           <section>
             <SectionHeader title="PLAYBACK CONTROLS" />
             <div className="grid grid-cols-2 gap-3">
               <ToggleOption 
                 label="AUTOPLAY" 
                 active={config.autoplay} 
                 onClick={() => setConfig(prev => ({ ...prev, autoplay: !prev.autoplay }))}
                 icon={IconPlay}
               />
               <ToggleOption 
                 label="MUTED" 
                 active={config.muted} 
                 onClick={() => setConfig(prev => ({ ...prev, muted: !prev.muted }))}
                 icon={IconMute}
               />
               <ToggleOption 
                 label="LOOP" 
                 active={config.loop} 
                 onClick={() => setConfig(prev => ({ ...prev, loop: !prev.loop }))}
                 icon={IconRepeat}
               />
               <ToggleOption 
                 label="CORS PROXY" 
                 active={config.proxy} 
                 onClick={() => setConfig(prev => ({ ...prev, proxy: !prev.proxy }))}
                 icon={IconGlobe}
                 warning
               />
               <ToggleOption 
                 label="NATIVE UI" 
                 active={config.controls} 
                 onClick={() => setConfig(prev => ({ ...prev, controls: !prev.controls }))}
                 icon={IconSettings}
               />
             </div>
           </section>

          </div>
        </div>
      </div>

      {/* --- Bottom Panel: Local + Export --- */}
      <div className="h-[38vh] min-h-[260px] bg-void-panel border-t border-void-border shadow-2xl z-20">
        <div className="h-full overflow-y-auto overflow-x-hidden p-6 custom-scrollbar">
          <div className="grid gap-6 xl:grid-cols-[minmax(0,1fr)_minmax(0,420px)]">
            <div className="space-y-6 order-2 xl:order-1">
              <section className="bg-void-black/30 rounded-xl border border-void-border p-4">
                <div className="flex justify-between items-center mb-4">
                  <SectionHeader title="LOCAL VOD" />
                  <div className={`text-[10px] font-bold px-2 py-0.5 rounded-full ${localConfig.enabled ? 'bg-accent/10 text-accent' : 'bg-red-500/10 text-red-500'}`}>
                    {localConfig.enabled ? 'ONLINE' : 'OFFLINE'}
                  </div>
                </div>

                {localConfig.enabled && (
                  <>
                    {adminStatus === 'guest' && (
                      <div className="space-y-3">
                        <input
                          className="w-full bg-void-black border border-void-border rounded px-3 py-2 text-sm"
                          value={loginForm.username}
                          onChange={(e) => setLoginForm((prev) => ({ ...prev, username: e.target.value }))}
                          placeholder="ADMIN_USER"
                        />
                        <input
                          type="password"
                          className="w-full bg-void-black border border-void-border rounded px-3 py-2 text-sm"
                          value={loginForm.password}
                          onChange={(e) => setLoginForm((prev) => ({ ...prev, password: e.target.value }))}
                          placeholder="ADMIN_PASS"
                        />
                        <button
                          onClick={handleLogin}
                          className="w-full py-2 bg-accent hover:bg-accent-hover text-black font-bold text-xs rounded shadow-[0_0_10px_rgba(255,217,0,0.3)] transition"
                        >
                          AUTHENTICATE
                        </button>
                        {adminError && <div className="text-red-500 text-xs text-center font-mono">{adminError}</div>}
                      </div>
                    )}

                    {adminStatus === 'authenticated' && (
                      <div className="space-y-4">
                        <div className="grid grid-cols-2 gap-2">
                          <button
                            onClick={() => openBrowse(browseState.dir)}
                            className="flex items-center justify-center gap-2 py-2 bg-void-border hover:bg-zinc-700 text-zinc-200 text-xs font-bold rounded border border-void-border transition"
                          >
                            <IconFolder /> BROWSE
                          </button>
                          <button
                            onClick={handleLogout}
                            className="flex items-center justify-center gap-2 py-2 bg-void-border hover:bg-red-900/30 hover:border-red-900/50 hover:text-red-400 text-zinc-400 text-xs font-bold rounded border border-void-border transition"
                          >
                            LOGOUT
                          </button>
                        </div>

                        <div className="p-3 bg-void-black rounded border border-void-border">
                          <div className="flex items-center gap-2 mb-2">
                            <input
                              type="number"
                              min={1}
                              className="bg-void-panel w-16 text-center border border-void-border rounded text-xs py-1"
                              value={shareTtlHours}
                              onChange={(e) => setShareTtlHours(Number(e.target.value))}
                            />
                            <span className="text-xs text-zinc-500 font-bold">HOURS TTL</span>
                          </div>
                          <button
                            onClick={handleCreateShare}
                            disabled={isShareBusy}
                            className="w-full py-2 bg-zinc-100 hover:bg-white text-black font-bold text-xs rounded transition"
                          >
                            {isShareBusy ? 'GENERATING...' : 'CREATE SHARE LINK'}
                          </button>
                        </div>

                        {localError && <div className="text-red-500 text-xs font-mono">{localError}</div>}

                        <div className="space-y-2">
                          <div className="text-[10px] font-bold text-zinc-500 uppercase">Active Shares</div>
                          {shares.length === 0 && <div className="text-zinc-600 text-xs italic">No active shares.</div>}
                          <div className="space-y-2 max-h-40 overflow-y-auto pr-1">
                            {shares.map((share) => {
                              const isActive = activeShareId === share.shareId
                              const isPosterBusy = sharePosterBusyId === share.shareId
                              return (
                                <div
                                  key={share.shareId}
                                  className={`bg-void-black/50 p-2 rounded border flex justify-between items-start group ${isActive ? 'border-accent/40 bg-accent/5' : 'border-void-border'}`}
                                >
                                  <div className="min-w-0 flex-1 mr-2">
                                    <div className="text-xs text-zinc-300 truncate font-mono" title={share.path}>{share.path}</div>
                                    <div className="flex items-center gap-2 text-[10px] text-zinc-600 mt-1">
                                      <span>Exp: {new Date(share.expiresAt * 1000).toLocaleTimeString()}</span>
                                      {isActive && (
                                        <span className="bg-accent/20 text-accent px-1.5 py-0.5 rounded">USED</span>
                                      )}
                                      {isPosterBusy && (
                                        <span className="text-zinc-400">UPDATING...</span>
                                      )}
                                    </div>
                                  </div>
                                  <div className="flex flex-col gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
                                    <button
                                      onClick={() => handleUseShare(share)}
                                      disabled={isActive}
                                      className={`text-[10px] px-2 py-1 rounded ${isActive ? 'bg-accent/10 text-accent/60 cursor-default' : 'bg-accent/20 text-accent hover:bg-accent/30'}`}
                                    >
                                      {isActive ? 'USING' : 'USE'}
                                    </button>
                                    <button
                                      onClick={() => {
                                        setSharePosterTarget(share)
                                        handleOpenUploads()
                                      }}
                                      disabled={isPosterBusy}
                                      className={`text-[10px] px-2 py-1 rounded ${isPosterBusy ? 'bg-void-border text-zinc-500 cursor-default' : 'bg-void-border text-zinc-200 hover:bg-zinc-700'}`}
                                    >
                                      {isPosterBusy ? '...' : 'POSTER'}
                                    </button>
                                    <button onClick={() => handleRevokeShare(share.shareId)} className="text-[10px] bg-red-900/30 text-red-400 px-2 py-1 rounded hover:bg-red-900/50">DEL</button>
                                  </div>
                                </div>
                              )
                            })}
                          </div>
                        </div>

                        <div className="space-y-3 border-t border-void-border/80 pt-4">
                          <div className="flex items-center justify-between">
                            <div className="text-[10px] font-bold text-zinc-500 uppercase">Embed Tokens</div>
                            <button
                              onClick={loadEmbedTokens}
                              className="text-[10px] text-zinc-400 hover:text-white"
                            >
                              Refresh
                            </button>
                          </div>

                          <div className="space-y-2">
                            <label className="text-[10px] font-bold text-zinc-500">Description</label>
                            <input
                              className="w-full bg-void-black border border-void-border rounded px-3 py-2 text-xs text-white"
                              placeholder="Optional label"
                              value={embedTokenDescription}
                              onChange={(event) => setEmbedTokenDescription(event.target.value)}
                            />
                          </div>

                          <div className="space-y-2">
                            <label className="text-[10px] font-bold text-zinc-500">TTL</label>
                            <div className="flex gap-2">
                              <button
                                onClick={() => setEmbedTtlMode('preset')}
                                className={`px-3 py-1 rounded text-[10px] font-bold border ${embedTtlMode === 'preset' ? 'bg-accent/20 text-accent border-accent' : 'bg-transparent text-zinc-500 border-void-border'}`}
                              >
                                PRESET
                              </button>
                              <button
                                onClick={() => setEmbedTtlMode('custom')}
                                className={`px-3 py-1 rounded text-[10px] font-bold border ${embedTtlMode === 'custom' ? 'bg-accent/20 text-accent border-accent' : 'bg-transparent text-zinc-500 border-void-border'}`}
                              >
                                CUSTOM
                              </button>
                            </div>

                            {embedTtlMode === 'preset' ? (
                              <div className="space-y-2">
                                <div className="text-[10px] text-zinc-500">Hours</div>
                                <div className="flex flex-wrap gap-2">
                                  {embedTtlPresets.hours.map((preset) => (
                                    <button
                                      key={preset}
                                      onClick={() => setEmbedTtlPreset(preset)}
                                      className={`px-2 py-1 rounded text-[10px] font-bold border ${embedTtlPreset === preset ? 'bg-accent/20 text-accent border-accent' : 'bg-transparent text-zinc-500 border-void-border hover:border-zinc-500'}`}
                                    >
                                      {preset}
                                    </button>
                                  ))}
                                </div>
                                <div className="text-[10px] text-zinc-500">Days</div>
                                <div className="flex flex-wrap gap-2">
                                  {embedTtlPresets.days.map((preset) => (
                                    <button
                                      key={preset}
                                      onClick={() => setEmbedTtlPreset(preset)}
                                      className={`px-2 py-1 rounded text-[10px] font-bold border ${embedTtlPreset === preset ? 'bg-accent/20 text-accent border-accent' : 'bg-transparent text-zinc-500 border-void-border hover:border-zinc-500'}`}
                                    >
                                      {preset}
                                    </button>
                                  ))}
                                </div>
                              </div>
                            ) : (
                              <div className="flex items-center gap-2">
                                <input
                                  type="number"
                                  min={1}
                                  className="bg-void-panel w-16 text-center border border-void-border rounded text-xs py-1"
                                  value={embedCustomTtlValue}
                                  onChange={(event) => setEmbedCustomTtlValue(Number(event.target.value))}
                                />
                                <select
                                  className="bg-void-panel border border-void-border rounded text-xs px-2 py-1"
                                  value={embedCustomTtlUnit}
                                  onChange={(event) => setEmbedCustomTtlUnit(event.target.value as 'hours' | 'days')}
                                >
                                  <option value="hours">hours</option>
                                  <option value="days">days</option>
                                </select>
                              </div>
                            )}
                          </div>

                          <button
                            onClick={handleCreateEmbedToken}
                            disabled={embedCreateBusy}
                            className="w-full py-2 bg-accent/90 hover:bg-accent text-black font-bold text-xs rounded transition"
                          >
                            {embedCreateBusy ? 'CREATING...' : 'CREATE EMBED TOKEN'}
                          </button>

                          {embedTokensError && <div className="text-red-500 text-xs font-mono">{embedTokensError}</div>}

                          <div className="space-y-2">
                            <div className="text-[10px] font-bold text-zinc-500 uppercase">Active Tokens</div>
                            {embedTokensLoading && <div className="text-zinc-600 text-xs italic">Loading...</div>}
                            {!embedTokensLoading && embedTokens.length === 0 && (
                              <div className="text-zinc-600 text-xs italic">No embed tokens yet.</div>
                            )}
                            <div className="space-y-2 max-h-40 overflow-y-auto pr-1">
                              {embedTokens.map((token) => {
                                const isSelected = selectedEmbedTokenId === token.tokenId
                                return (
                                  <div
                                    key={token.tokenId}
                                    className={`bg-void-black/50 p-2 rounded border flex justify-between items-start group ${isSelected ? 'border-accent/40 bg-accent/5' : 'border-void-border'}`}
                                  >
                                    <div className="min-w-0 flex-1 mr-2">
                                      <div className="text-xs text-zinc-300 truncate font-mono" title={token.tokenId}>{token.description || token.tokenId}</div>
                                      <div className="flex flex-wrap items-center gap-2 text-[10px] text-zinc-600 mt-1">
                                        <span>Exp: {token.expiresAt ? new Date(token.expiresAt * 1000).toLocaleString() : 'Never'}</span>
                                        <span>Uses: {token.usageCount || 0}</span>
                                        <span>Last: {token.lastAccessedAt ? new Date(token.lastAccessedAt * 1000).toLocaleString() : '—'}</span>
                                      </div>
                                    </div>
                                    <div className="flex flex-col gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
                                      <button
                                        onClick={() => setSelectedEmbedTokenId(token.tokenId)}
                                        disabled={isSelected}
                                        className={`text-[10px] px-2 py-1 rounded ${isSelected ? 'bg-accent/10 text-accent/60 cursor-default' : 'bg-accent/20 text-accent hover:bg-accent/30'}`}
                                      >
                                        {isSelected ? 'SELECTED' : 'USE'}
                                      </button>
                                      <button
                                        onClick={() => handleRevokeEmbedToken(token.tokenId)}
                                        className="text-[10px] bg-red-900/30 text-red-400 px-2 py-1 rounded hover:bg-red-900/50"
                                      >
                                        REVOKE
                                      </button>
                                    </div>
                                  </div>
                                )
                              })}
                            </div>
                          </div>
                        </div>
                      </div>
                    )}
                  </>
                )}
              </section>
            </div>

            <div className="space-y-6 order-1 xl:order-2">
              <section className="bg-void-black/20 rounded-xl border border-void-border p-4">
                <SectionHeader title="CONFIGURATION" />
                <div className="space-y-4">
                  <div>
                    <label className="text-[10px] font-bold text-zinc-500 mb-1.5 block">POSTER IMAGE</label>
                    <div className="flex flex-wrap gap-2">
                      <input
                        className="flex-1 min-w-[220px] bg-void-black border border-void-border rounded px-3 py-2 text-xs text-white placeholder-zinc-700 focus:outline-none focus:border-zinc-600 font-mono"
                        placeholder="https://..."
                        value={config.poster}
                        onChange={(e) => setConfig((prev) => ({ ...prev, poster: e.target.value }))}
                      />
                      <input
                        type="file"
                        ref={fileInputRef}
                        onChange={handleFileUpload}
                        className="hidden"
                        accept="image/*"
                      />
                      <button
                        onClick={() => fileInputRef.current?.click()}
                        disabled={isUploading}
                        className="px-3 py-2 bg-void-border hover:bg-zinc-700 text-zinc-300 text-xs font-bold rounded border border-void-border transition"
                      >
                        {isUploading ? '...' : 'UPLOAD'}
                      </button>
                      <button
                        onClick={handleOpenUploads}
                        className="px-3 py-2 bg-void-border hover:bg-zinc-700 text-zinc-300 text-xs font-bold rounded border border-void-border transition"
                      >
                        BROWSE
                      </button>
                    </div>
                  </div>

                  <div>
                    <label className="text-[10px] font-bold text-zinc-500 mb-1.5 block">PRELOAD STRATEGY</label>
                    <select
                      className="w-full bg-void-black border border-void-border rounded px-3 py-2 text-xs text-zinc-300 focus:outline-none focus:border-zinc-600"
                      value={config.preload}
                      onChange={(e) => setConfig((prev) => ({ ...prev, preload: e.target.value as PlayerConfig['preload'] }))}
                    >
                      <option value="metadata">METADATA (RECOMMENDED)</option>
                      <option value="auto">AUTO</option>
                      <option value="none">NONE</option>
                    </select>
                  </div>
                </div>
              </section>

              <section className="border border-void-border bg-void-panel/90 backdrop-blur rounded-xl p-4">
                {!showEmbedCode ? (
                  <button
                    aria-label="export_embed"
                    onClick={() => setShowEmbedCode(true)}
                    className="w-full py-3 bg-void-border hover:bg-zinc-700 text-zinc-300 font-bold text-xs rounded border border-void-border flex items-center justify-center gap-2 transition"
                  >
                    <IconCode /> EXPORT EMBED CODE
                  </button>
                ) : (
                  <div className="animate-slide-down space-y-3">
                    <div className="flex justify-between items-center">
                      <span className="text-xs font-bold text-zinc-400">
                        EMBED SNIPPET
                        <span className="sr-only">EMBED_CODE</span>
                      </span>
                      <button onClick={() => setShowEmbedCode(false)} className="text-xs text-zinc-500 hover:text-white">CLOSE</button>
                    </div>

                    <div className="flex gap-2 mb-2">
                      {embedSizeOptions.map((size) => (
                        <button
                          key={size}
                          onClick={() => setEmbedSize(size)}
                          className={`px-3 py-1 rounded text-[10px] font-bold border transition ${embedSize === size ? 'bg-accent/20 text-accent border-accent' : 'bg-transparent text-zinc-500 border-void-border hover:border-zinc-500'}`}
                        >
                          {size === 'responsive' ? 'RESPONSIVE' : size === 'fixed-medium' ? '640px' : '480px'}
                        </button>
                      ))}
                    </div>

                    <div className="relative group">
                      <pre
                        onClick={handleCopy}
                        className="bg-black border border-void-border rounded p-3 text-[10px] text-zinc-400 font-mono overflow-x-auto whitespace-pre-wrap word-break-all cursor-pointer hover:border-accent/50 hover:text-zinc-300 transition"
                      >
                        {embedSnippet}
                      </pre>
                      {copyFeedback && (
                        <div className="absolute inset-0 flex items-center justify-center bg-accent/90 text-black text-xs font-bold rounded backdrop-blur">
                          COPIED TO CLIPBOARD!
                        </div>
                      )}
                    </div>
                  </div>
                )}
              </section>
            </div>
          </div>
        </div>
      </div>
      
      {/* Modals */}
      <FileBrowserDialog 
        isOpen={browseState.open}
        dir={browseState.dir}
        loading={browseState.loading}
        error={browseState.error || null}
        entries={browseState.entries}
        onClose={() => setBrowseState(prev => ({ ...prev, open: false }))}
        onNavigate={openBrowse}
        onSelect={selectBrowseEntry}
      />
      <UploadedPostersDialog
        isOpen={uploadsOpen}
        entries={uploadedPosters}
        loading={uploadsLoading}
        error={uploadsError}
        onClose={handleCloseUploads}
        onSelect={handleSelectUpload}
        onRefresh={loadUploads}
      />
    </div>
  )
}
