#!/usr/bin/env bash
set -euo pipefail

ROOT_DIR=$(cd "$(dirname "$0")" && pwd)
ENV_FILE="$ROOT_DIR/hls-player/.env"

# Ignore any externally-exported values; we only trust .env or defaults.
unset FRONTEND_PORT BACKEND_PORT FRONTEND_HOST BACKEND_HOST FRONTEND_PUBLIC_URL \
  BACKEND_PUBLIC_URL PRIOMPT_PREVIEW_SERVER_PORT PORT HOST

if [ -f "$ENV_FILE" ]; then
  set -a
  . "$ENV_FILE"
  set +a
fi

FRONTEND_PORT=${FRONTEND_PORT:-7594}
BACKEND_PORT=${BACKEND_PORT:-2738}
FRONTEND_HOST=${FRONTEND_HOST:-0.0.0.0}
BACKEND_HOST=${BACKEND_HOST:-0.0.0.0}
FRONTEND_PUBLIC_URL=${FRONTEND_PUBLIC_URL:-"http://127.0.0.1:${FRONTEND_PORT}"}
BACKEND_PUBLIC_URL=${BACKEND_PUBLIC_URL:-"http://127.0.0.1:${BACKEND_PORT}"}
PRIOMPT_PREVIEW_SERVER_PORT=${PRIOMPT_PREVIEW_SERVER_PORT:-$FRONTEND_PORT}
LOG_DIR="$ROOT_DIR/logs"
LOG_FILE="$LOG_DIR/prod.log"

mkdir -p "$LOG_DIR"
: > "$LOG_FILE"

# Redirect output to both log file and original stdout/stderr
# We use a named pipe approach to allow simultaneous logging and display
exec > >(tee -a "$LOG_FILE") 2>&1

validate_port() {
  local value="$1"
  local label="$2"
  if ! [[ "$value" =~ ^[0-9]+$ ]]; then
    echo "Invalid ${label}: ${value}" >&2
    exit 1
  fi
  if [ "$value" -lt 1 ] || [ "$value" -gt 65535 ]; then
    echo "Invalid ${label}: ${value} (out of range)" >&2
    exit 1
  fi
}

validate_port "$FRONTEND_PORT" "FRONTEND_PORT"
validate_port "$BACKEND_PORT" "BACKEND_PORT"
validate_port "$PRIOMPT_PREVIEW_SERVER_PORT" "PRIOMPT_PREVIEW_SERVER_PORT"

echo "prod.sh config:"
echo "  FRONTEND_HOST=$FRONTEND_HOST"
echo "  FRONTEND_PORT=$FRONTEND_PORT"
echo "  BACKEND_HOST=$BACKEND_HOST"
echo "  BACKEND_PORT=$BACKEND_PORT"
echo "  FRONTEND_PUBLIC_URL=$FRONTEND_PUBLIC_URL"
echo "  BACKEND_PUBLIC_URL=$BACKEND_PUBLIC_URL"
echo "  PRIOMPT_PREVIEW_SERVER_PORT=$PRIOMPT_PREVIEW_SERVER_PORT"

install_frontend() {
  echo "Installing frontend dependencies..."
  cd "$ROOT_DIR/hls-player"
  if [ -d "$ROOT_DIR/hls-player/node_modules" ]; then
    rm -rf "$ROOT_DIR/hls-player/node_modules"
  fi
  npm ci --no-audit --prefer-offline
}

install_backend() {
  echo "Installing backend dependencies..."
  cd "$ROOT_DIR/backend"
  if [ -d "$ROOT_DIR/backend/node_modules" ]; then
    rm -rf "$ROOT_DIR/backend/node_modules"
  fi
  npm ci --omit=dev --no-audit --prefer-offline
}

install_frontend
echo "Building frontend..."
(
  cd "$ROOT_DIR/hls-player"
  # Ensure VITE_BACKEND_URL is available at build time
  VITE_BACKEND_URL="$BACKEND_PUBLIC_URL" \
  NODE_ENV=production npm run build
)
install_backend

echo "Starting backend on port $BACKEND_PORT..."
(
  cd "$ROOT_DIR/backend"
  BACKEND_HOST="$BACKEND_HOST" \
  BACKEND_PORT="$BACKEND_PORT" \
  FRONTEND_URL="$FRONTEND_PUBLIC_URL" \
  PRIOMPT_PREVIEW_SERVER_PORT="$PRIOMPT_PREVIEW_SERVER_PORT" \
  NODE_ENV=production npm run start
) &
BACKEND_PID=$!

echo "Starting frontend on port $FRONTEND_PORT..."
(
  cd "$ROOT_DIR/hls-player"
  # Use 'vite preview' instead of 'npx serve' for better reliability and consistency with dev environment.
  # We use --strictPort to ensure it fails rather than switching if the port is taken.
  npm run preview -- --port "$FRONTEND_PORT" --host "$FRONTEND_HOST" --strictPort
) &
FRONTEND_PID=$!

cleanup() {
  echo "Stopping services..."
  kill "$FRONTEND_PID" "$BACKEND_PID" 2>/dev/null || true
}
trap cleanup EXIT

wait "$FRONTEND_PID" "$BACKEND_PID"
